//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BitmapBrushColorBlend.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(BitmapBrushColorBlend, CombinedEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    BitmapBrushColorBlend::BitmapBrushColorBlend()
    {
    }

    BitmapBrushColorBlend::BitmapBrushColorBlend(const BitmapBrushColorBlend& rhs) :
        Base(rhs),
        m_bitmapBrush(rhs.m_bitmapBrush),
        m_colorEffect(rhs.m_colorEffect),
        m_blendEffect(rhs.m_blendEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    BitmapBrushColorBlend::~BitmapBrushColorBlend()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    BitmapBrushColorBlend::SharedPointer BitmapBrushColorBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        BitmapBrushColorBlend* brush = FEATSTD_NEW(BitmapBrushColorBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        BitmapBrushColorBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void BitmapBrushColorBlend::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform,
                                                 const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "BitmapBrushColorBlend"));

        if (m_bitmapBrush.Image() != 0) {
            m_blendEffect.ActivateBlend(output);
            Color::Data& color = m_colorEffect.Color().Get();
            Float cameraAlpha = 1.0F;
            const Camera2D* camera2D = Renderer2D::GetActiveCamera();
            if (0 != camera2D) {
                if (camera2D->IsCameraEffectiveAlphaEnabled()) {
                    cameraAlpha = camera2D->GetEffectiveAlphaValue();
                }
            }

            bool result = RenderDevice2D::SetSurfaceConstColor(
                output,
                RenderDevice2D::SourceSurface,
                color.red,
                color.green,
                color.blue,
                color.alpha * node.GetEffectiveAlphaValue() * cameraAlpha);
            if (result ) {
                m_bitmapBrush.Render(input, inputArea, transform, node, output, outputArea);
                static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.F, 1.F, 1.F, 1.F));
            }
            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool BitmapBrushColorBlend::Upload()
    {
        bool success = m_bitmapBrush.Upload();
        success = m_colorEffect.Upload() && success;
        success = m_blendEffect.Upload() && success;
        return success;
    }
    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool BitmapBrushColorBlend::Unload()
    {
        bool success = m_bitmapBrush.Unload();
        success = m_colorEffect.Unload() && success;
        success = m_blendEffect.Unload() && success;
        return success;
    }
    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool BitmapBrushColorBlend::Update()
    {
        bool success = m_bitmapBrush.Update();
        success = m_colorEffect.Update() && success;
        success = m_blendEffect.Update() && success;
        return success;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer BitmapBrushColorBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(BitmapBrushColorBlend)(*this));
    }

}   // namespace Candera
