//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BitmapBrushColor.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(BitmapBrushColor, CombinedEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    BitmapBrushColor::BitmapBrushColor()
    {
    }

    BitmapBrushColor::BitmapBrushColor(const BitmapBrushColor& rhs) :
        Base(rhs),
        m_bitmapBrush(rhs.m_bitmapBrush),
        m_colorEffect(rhs.m_colorEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    BitmapBrushColor::~BitmapBrushColor()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    BitmapBrushColor::SharedPointer BitmapBrushColor::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        BitmapBrushColor* brush = FEATSTD_NEW(BitmapBrushColor);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        BitmapBrushColor::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void BitmapBrushColor::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node,
                                       ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "BitmapBrushColor"));

        if (m_bitmapBrush.Image() != 0) {
            Color::Data& color = m_colorEffect.Color().Get();
            Float cameraAlpha = 1.0F;
            const Camera2D* camera2D = Renderer2D::GetActiveCamera();
            if (0 != camera2D) {
                if (camera2D->IsCameraEffectiveAlphaEnabled()) {
                    cameraAlpha = camera2D->GetEffectiveAlphaValue();
                }
            }
            bool result = RenderDevice2D::SetSurfaceConstColor(
                output,
                RenderDevice2D::SourceSurface,
                color.red,
                color.green,
                color.blue,
                color.alpha * node.GetEffectiveAlphaValue() * cameraAlpha);
            if (result) {
                m_bitmapBrush.Render(input, inputArea, transform, node, output, outputArea);
                static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.F, 1.F, 1.F, 1.F));
            }
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool BitmapBrushColor::Upload()
    {
        bool success = m_bitmapBrush.Upload();
        success = m_colorEffect.Upload() && success;
        return success;
    }
    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool BitmapBrushColor::Unload()
    {
        bool success = m_bitmapBrush.Unload();
        success = m_colorEffect.Unload() && success;
        return success;
    }
    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool BitmapBrushColor::Update()
    {
        bool success = m_bitmapBrush.Update();
        success = m_colorEffect.Update() && success;
        return success;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer BitmapBrushColor::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(BitmapBrushColor)(*this));
    }

}   // namespace Candera
