//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(EGL_WRAPPER_H)
#define EGL_WRAPPER_H

#include <Candera/Environment.h>
#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <CanderaPlatform/OS/CanderaTypes.h>


namespace Candera { 
    // Forward declaration.
    namespace UnitTests {
        class EGLWrapperTest;
    }
    namespace Internal {
/** @addtogroup CommonDevice
*  @{
*/
    
    /**
    *  @brief This class provides a state caching mechanism for EGL interfaces which are more CPU consuming.
    *         The state caching feature can be turned on/off from Cmake configuration flag CANDERA_RENDER_STATE_CACHING_ENABLED.
    *         When state caching is turned oFf, this class behaves as a simple wrapper which forwards API's call directly to EGL layer.
    */
    class EGLWrapper
    {
        friend class UnitTests::EGLWrapperTest;

    public:
        /**
        * Forwards API call to eglMakeCurrent(). The call to EGL is made only if the context (function parameters) where
        * now cached before.
        * @param dpy Points to display
        * @param draw Points to EGL destination surface memory
        * @param read Points to EGL read surface source
        * @param ctx  Points to the current EGL Context mem location.
        * @return  status of call, success or fail.
        */
        EGLBoolean MakeCurrent(EGLDisplay dpy, EGLSurface draw, EGLSurface read, EGLContext ctx);

        /**
        * Returns the EGL context from cache or from EGL layer.
        * @return  pointer to EGL context.
        */
        EGLContext GetCurrentContext() const;

        /**
        * Returns the EGL's current display from cache or from EGL layer.
        * @return  pointer to EGL display.
        */
        EGLDisplay GetCurrentDisplay() const;

        /**
        * Returns the EGL's current read OR draw surface.
        * @parameter readdraw Specify which surface to return.
        * @return  pointer to EGL surface.
        */
        EGLSurface GetCurrentSurface(EGLint readdraw) const;

        /**
        * Implements a single ton design pattern.
        * @return  Reference to static instance of this class
        */
        static EGLWrapper& GetInstance();

        /**
        * Destructor
        */
        ~EGLWrapper() { /* Destructor */ }
    protected:

    private:
        /**
        * Constructor
        */
        EGLWrapper();

#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED    

        struct  PackEGLMakeCurrentParameters
        {
            EGLDisplay    m_dpy;
            EGLSurface    m_draw;
            EGLSurface    m_read;
            EGLContext    m_ctx;
            bool          m_bCacheIsValid;

            PackEGLMakeCurrentParameters() :
                m_dpy(EGL_NO_DISPLAY),
                m_draw(EGL_NO_SURFACE),
                m_read(EGL_NO_SURFACE),
                m_ctx(EGL_NO_CONTEXT),
                m_bCacheIsValid(false)
            {}
        };

        /**
        * Private data section used only when caching feature is enabled from cmake.
        */

        PackEGLMakeCurrentParameters m_apiMakeCurrentParams;

        /**
        * Copy function parameters from eglMakeCurrent() to local cache.
        * @param dpy Points to display
        * @param draw Points to EGL destination surface memory
        * @param read Points to EGL read surface source
        * @param ctx  Points to the current EGL Context mem location.
        */
        void CacheMakeCurrentParameters(EGLDisplay dpy, EGLSurface draw, EGLSurface read, EGLContext ctx);

        /**
        * This function allways frowards the call for eglMakeCurrent() to EGL layer.
        * if the returned state is TRUE, then the parameters are also cached locally.
        * @param dpy Points to display
        * @param draw Points to EGL destination surface memory
        * @param read Points to EGL read surface source
        * @param ctx  Points to the current EGL Context memory location.
        * @return  TRUE or FALSE
        */
        EGLBoolean FwdCall2EGLMakeCurrent(EGLDisplay dpy, EGLSurface draw, EGLSurface read, EGLContext ctx);
#endif
    };
        /** @}*/ //end of CommonDevice
} /* namespace Internal */
} /* namespace Candera */
#endif /* EGL_WRAPPER_H */
