//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "EglWrapper.h"

namespace Candera { 
    namespace Internal {

        EGLWrapper::EGLWrapper()
#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED
            : m_apiMakeCurrentParams()
#endif
        {
            /* Constructor */
        }

        EGLWrapper& EGLWrapper::GetInstance()
        {
            static EGLWrapper m_instance;
            return m_instance;
        }

        EGLBoolean EGLWrapper::MakeCurrent(EGLDisplay dpy, EGLSurface draw, EGLSurface read, EGLContext ctx)
        {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::Internal::EGLWrapper::MakeCurrent, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects depending on compile switch and is therefore considered non-const")
            EGLBoolean bResult = EGL_FALSE;
#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED
            if (m_apiMakeCurrentParams.m_bCacheIsValid) {
                // Cache is valid, now check if context is different.
                if ((m_apiMakeCurrentParams.m_dpy != dpy) || (m_apiMakeCurrentParams.m_draw != draw)
                    || (m_apiMakeCurrentParams.m_read != read) || (m_apiMakeCurrentParams.m_ctx != ctx)) {

                    // Context is different so forward this call to EGL layer.
                    bResult = FwdCall2EGLMakeCurrent(dpy, draw, read, ctx);
                }
                else {
                    // No need to forward call to EGL layer, just return TRUE as the previous call did.
                    bResult = EGL_TRUE;
                }
            }
            else {
                // Cache is invalid, so always forward the call to EGL layer.
                bResult = FwdCall2EGLMakeCurrent(dpy, draw, read, ctx);
            }
#else
            // Caching feature is not activated, so always forward this call to EGL layer.
            FEATSTD_LINT_NEXT_EXPRESSION(838, "Violates MISRA C++ 2008 Required Rule 0-1-9: previously assigned value")
            bResult = eglMakeCurrent(dpy, draw, read, ctx);
#endif
            return bResult;
        }

        EGLContext EGLWrapper::GetCurrentContext(void) const
        {
            EGLContext context = EGL_NO_CONTEXT;
#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED
            if (m_apiMakeCurrentParams.m_bCacheIsValid) {
                // Valid cache, retrieved the cached value.
                context = m_apiMakeCurrentParams.m_ctx;
            }
            else {
                // invalid cache, forward this call to EGL layer.
                context = eglGetCurrentContext();
            }
#else
            // Caching feature is not activated, so always forward this call to EGL layer.
            FEATSTD_LINT_NEXT_EXPRESSION(838, "Violates MISRA C++ 2008 Required Rule 0-1-9: previously assigned value")
            context = eglGetCurrentContext();
#endif

            return context;
        }

        EGLDisplay EGLWrapper::GetCurrentDisplay() const
        {
            EGLDisplay display = EGL_NO_DISPLAY;
#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED
            if (m_apiMakeCurrentParams.m_bCacheIsValid) {
                // Valid cache, retrieved the cached display value.
                display = m_apiMakeCurrentParams.m_dpy;
            }
            else {
                // invalid cache, forward this call to EGL layer.
                display = eglGetCurrentDisplay();
            }
#else
            // Caching feature is not activated, so always forward this call to EGL layer.
            FEATSTD_LINT_NEXT_EXPRESSION(838, "Violates MISRA C++ 2008 Required Rule 0-1-9: previously assigned value")
            display = eglGetCurrentDisplay();
#endif

            return display;
        }

        EGLSurface EGLWrapper::GetCurrentSurface(EGLint readdraw) const
        {
            EGLSurface surface = EGL_NO_SURFACE;

#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED
            if (m_apiMakeCurrentParams.m_bCacheIsValid) {
                // Valid cache, retrieved the cached display value.
                switch (readdraw)
                {
                case EGL_READ: surface = m_apiMakeCurrentParams.m_read; break;
                case EGL_DRAW: surface = m_apiMakeCurrentParams.m_draw; break;
                default:  surface = EGL_NO_SURFACE;                     break;
                }
            }
            else {
                // invalid cache, forward this call to EGL layer.
                surface = eglGetCurrentSurface(readdraw);
            }
#else
            // Caching feature is not activated, so always forward this call to EGL layer.
            FEATSTD_LINT_NEXT_EXPRESSION(838, "Violates MISRA C++ 2008 Required Rule 0-1-9: previously assigned value")
            surface = eglGetCurrentSurface(readdraw);
#endif
            return surface;
        }

#ifdef CANDERA_RENDER_STATE_CACHING_ENABLED
        void EGLWrapper::CacheMakeCurrentParameters(EGLDisplay dpy, EGLSurface draw, EGLSurface read, EGLContext ctx)
        {
            m_apiMakeCurrentParams.m_dpy = dpy;
            m_apiMakeCurrentParams.m_draw = draw;
            m_apiMakeCurrentParams.m_read = read;
            m_apiMakeCurrentParams.m_ctx = ctx;
        }

        EGLBoolean EGLWrapper::FwdCall2EGLMakeCurrent(EGLDisplay dpy, EGLSurface draw, EGLSurface read, EGLContext ctx)
        {
            EGLBoolean bResult = eglMakeCurrent(dpy, draw, read, ctx);
            if (EGL_TRUE == bResult) {
                // No error so, cache parameters
                this->CacheMakeCurrentParameters(dpy, draw, read, ctx);
                m_apiMakeCurrentParams.m_bCacheIsValid = true;
            }
            else {
                // Error occurred, do not cache parameters.
                m_apiMakeCurrentParams.m_bCacheIsValid = false;
            }

            return bResult;
        }
#endif

    } /* namespace Internal */
} /* namespace Candera */
