//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_EGLCONFIGURATION_H)
#define CANDERA_EGLCONFIGURATION_H

#include <CanderaPlatform/OS/CanderaTypes.h>

namespace Candera {
/** @addtogroup CommonDevice
*  @{
*/

    /**
     * @brief Abstraction of initial configuration for contexts and render targets.
     * EglConfiguration describes the format, tpye and size of color buffers and ancillary buffers for render targets.
     * A context can be used with any render target that is compatible with. A render target and context are compatibly if
     * - They support the same type of color buffer.
     * - They have color buffers and ancillary buffers of the same depth.
     * - The type of the supported rendering API is the same.
     * - They were created with respect to the same EglDisplay.
     * For more detail see Khronos EGL specification.
     */
    class EglConfiguration
    {
        public:
            enum EglConfigAttribute {
                EglBufferSize = 0,             ///< Depth of the color buffer.
                EglRedSize,                    ///< Bits of Red in the color buffer.
                EglGreenSize,                  ///< Bits of Green in the color buffer.
                EglBlueSize,                   ///< Bits of Blue in the color buffer.
                EglLuminanceSize,              ///< Bits of Luminance in the color buffer.
                EglAlphaSize,                  ///< Bits of Alpha in the color buffer.
                EglAlphaMaskSize,              ///< Bits of Alpha Mask in the color buffer.

                EglColorBufferType,            ///< Color buffer type, see enum ColorBufferType.
                EglConfigCaveat,               ///< Any caveat for the configuration, see enum Caveat.
                EglConfigId,                   ///< Unique EGLConfig identifier.
                EglConformant,                 ///< Bitmask indicating which client API contexts must be conformant, see enum ClientAPI.
                EglDepthSize,                  ///< Bits of Z in the depth buffer.
                EglLevel,                      ///< Frame buffer level.
                EglMaxSwapInterval,            ///< Maximum swap interval.
                EglMinSwapInterval,            ///< Minimum swap interval.
                EglNativeRenderable,           ///< Whether or not native rendering APIs can render to surface.
                EglNativeVisualType,           ///< Native visual type of the associated visual.
                EglRenderableType,             ///< Bitmask indicating which client APIs must be supported, see enum EglRenderableTypeValue.
                EglSampleBuffers,              ///< Number of multisample buffers.
                EglSamples,                    ///< Number of samples per pixel.
                EglStencilSize,                ///< Bits of Stencil in the stencil buffer.
                EglSurfaceType,                ///< Bitmask indicating which EGL surface types must be supported, see enum SurfaceType.
                EglTransparentType,            ///< Type of transparency supported, see enum TransparentType.
                EglTransparentRedValue,        ///< Transparent red value.
                EglTransparentGreenValue,      ///< Transparent green value.
                EglTransparentBlueValue,       ///< Transparent blue value.
                // Egl Extensions
                EglDepthEncoding,              ///< Encoding of depth buffer [linear or non-linear].
                EglCoverageSampleBuffers,      ///< Number of coverage buffers.
                EglCoverageSamples,            ///< Number of coverage samples per pixel.
                EglNumberOfConfigAttributes    ///< Number of Egl config attributes.
            };

            /*
            The following EglConfigAttributes are not supported:
                EglMaxPbufferWidth
                EglMaxPbufferHeight
                EglMaxPbufferPixels
                EglNativeVisualId
                EglMatchNativePixmap
                EglBindToTextureRgb           --> Covered by FramebufferRenderTarget
                EglBindToTextureRgba,         --> Covered by FramebufferRenderTarget
            */

            enum EglGeneralValue{
                EglDontCare = -1,                       ///< Don't care about the value of an attribute that has EglDontCare set.
                EglFalse = 0,                           ///< Equivalent to boolean false.
                EglTrue = 1                             ///< Equivalent to boolean true.
            };

            // enums

            enum EglColorBufferTypeValue {
                EglRgbBuffer = 0,                       ///< RGB color buffer.
                EglLumincanceBuffer                     ///< Luminance color buffer.
            };

            enum EglConfigCaveatValue {
                EglNoCaveat = 0,                        ///< Frame buffer configurations with no caveats will be considered.
                EglSlowCaveat,                          ///< Slow frame buffer configurations will be considered.
                EglNoneConformantCaveat                 ///< Non-conformant frame buffer configurations will be considered.
            };

            enum EglTransparentTypeValue {
                EglNoTransparency = 0,                  ///< No transparency.
                EglTransparentRgb                       ///< Allows to define one color to be transparent.
            };

            // Bitmasks

            enum EglConformantValue {
                EglConformantOpenGlBit = 0x1,           ///< OpenGL 1.x or 2.x must be supported.
                EglConformantOpenGlEsBit = 0x2,         ///< OpenGL ES 1.x must be supported.
                EglConformantOpenGlEs2Bit = 0x4,        ///< OpenGL ES 2.x must be supported.
                EglConformantOpenVgBit = 0x8            ///< OpenVG 1.x must be supported.
            };

            enum EglRenderableTypeValue {
                EglRenderableTypeOpenGlBit = 0x1,       ///< OpenGL 1.x or 2.x must be supported.
                EglRenderableTypeOpenGlEsBit = 0x2,     ///< OpenGL ES 1.x must be supported.
                EglRenderableTypeOpenGlEs2Bit = 0x4,    ///< OpenGL ES 2.x must be supported.
                EglRenderableTypeOpenVgBit = 0x8        ///< OpenVG 1.x must be supported.
            };

            enum EglSurfaceTypeValue {
                EglWindowBit = 0x1,                     ///< Windows must be supported.
                EglPixmapBit = 0x2,                     ///< Pixmaps must be supported.
                EglPBufferBit = 0x4,                    ///< PBuffers must be supported.
                EglMultisampleResolveBoxBit = 0x8,      ///< Box filtered multisample resolve must be supported.
                EglSwapBehaviourPreservedBit = 0x10,    ///< Swap behaviour for color buffers must be supported.
                EglVgColorspaceLinearBit = 0x20,        ///< OpenVG rendering in linear colorspace must be supported.
                EglVgAlphaFormatPreBit = 0x40           ///< OpenVG rendering with pre-multiplied alpha must be supported.
            };

            enum EglDepthEncodingValue {
                EglDepthEncodingNone = 0,               ///< No depth buffer encoding (linear).
                EglDepthEncodingNonlinear = 1           ///< Non-linear depth buffer is encoding.
            };


            /**
             * Constructor. Initialization of configuration with default values.
             */
            EglConfiguration();

            /**
             * Default Destructor.
             */
            ~EglConfiguration();

            /**
             * Gets the configuration attributes array.
             * Note: The size of the array can be obtained from EglConfiguration::EglNumberOfConfigAttributes.
             * @return Pointer that points to the configuration array.
             */
            const Int32* GetConfigurationAttributes() const { return m_configuration; }

            /**
             * Sets the value for a single configuration attribute.
             * @param attribute Attribute that should be updated.
             * @param value New value for the specified attribute.
             */
            void SetConfigurationAttribute(EglConfigAttribute attribute, Int32 value);
            Int32 GetConfigurationAttribute(EglConfigAttribute attribute) const;

        private:
            /// Array of configuration values.
            Int32 m_configuration[EglNumberOfConfigAttributes];
    };
    /** @}*/ //end of CommonDevice
} // namespace Candera

#endif    // CANDERA_EGLCONFIGURATION_H
