//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GENERIC_LINEAR_2D_DIRECT_ACCESSS_BITMAP_H)
#define GENERIC_LINEAR_2D_DIRECT_ACCESSS_BITMAP_H

#include <Candera/Environment.h>
#include <FeatStd/Diagnostics/Debug.h>

#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapConvertor.h>
#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapAccessInternal.h>

namespace Candera
{
    namespace GenericBitmapConvertor
    {
        //Class definitions
        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        class Linear2DDirectAccessContainer;

        /**
          * @brief Generic data accessor. Has 2 specializations: for bit level access and byte level access
          */
        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        class Linear2DDirectAccessor {};

        /**
          * @brief Iterator of a 2D Linear Bitmap with Direct Access to pixel colors
          */
        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        class Linear2DDirectAccessIterator
        {
        public:
            typedef Linear2DDirectAccessor<HostType, WordType, Endianness, bitLevelAccess> Accessor;

            /**
              * Constructor.
              * @param host container to interate through.
              * @param bufPos position on a raw.
              * @param line raw number.
              */
            Linear2DDirectAccessIterator(Linear2DDirectAccessContainer<HostType, WordType, Endianness, bitLevelAccess>& host, UInt32 bufPos, UInt32 line);

            /**
              * Increment the iterator to point to the next item within the container.
              * @return the iterator after incrementation
              */
            Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess> operator++ ();
            /**
              * Increment the iterator to point to the next item within the container.
              * @return the iterator before incrementation
              */
            Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess> operator++ (Internal::PostIncrementType);

            /**
              * Retrieve the item pointed to.
              * @return an item accessor.
              * @see Linear2DDirectAccessor.
              */
            Accessor operator* ();

            /**
              * Get the offset in bytes from the start of the container.
              * @return number of bytes from the start of the container.
              */
            UInt32 GetOffset() const;

            /**
              * Checkes whether the iterator has gone passed the end of the container.
              * @return true if the iterator has gone passed the end of the container.
              */
            bool IsAtEnd() const;

            friend class Linear2DDirectAccessor<HostType, WordType, Endianness, bitLevelAccess>;

        private:
            Linear2DDirectAccessContainer<HostType, WordType, Endianness, bitLevelAccess>& m_host;
            UInt32 m_bufPos;
            UInt32 m_line;

            //forbid assignment by making it private
            Linear2DDirectAccessIterator& operator=(const Linear2DDirectAccessIterator&);
        };

        /**
          * @brief Byte level data accessor
          */
        template <typename HostType, typename WordType, typename Endianness>
        class Linear2DDirectAccessor<HostType, WordType, Endianness, false>
        {
        public:
            /**
              * Read the item.
              * @return the value of the item packed within an integer.
              */
            const HostType get() const;
            /**
              * Write the item.
              * @param item the value of the item packed within an integer.
              */
            void set(HostType val);

            friend class Linear2DDirectAccessIterator<HostType, WordType, Endianness, false>;
        private:
            Linear2DDirectAccessor(Linear2DDirectAccessIterator<HostType, WordType, Endianness, false>& source);
            Linear2DDirectAccessIterator<HostType, WordType, Endianness, false>& m_source;

            //forbid assignment by making it private
            Linear2DDirectAccessor& operator=(const Linear2DDirectAccessor&);
        };

        /**
          * @brief Bit level data accessor
          */
        template <typename HostType, typename WordType, typename Endianness>
        class Linear2DDirectAccessor<HostType, WordType, Endianness, true>
        {
        public:
            /**
              * Read the item.
              * @return the value of the item packed within an integer.
              */
            const HostType get() const;
            /**
              * Write the item.
              * @param val the value of the item packed within an integer.
              */
            void set(HostType val);

            friend class Linear2DDirectAccessIterator<HostType, WordType, Endianness, true>;
        private:
            Linear2DDirectAccessor(Linear2DDirectAccessIterator<HostType, WordType, Endianness, true>& source);
            Linear2DDirectAccessIterator<HostType, WordType, Endianness, true>& m_source;

            //forbid assignment by making it private
            Linear2DDirectAccessor& operator=(const Linear2DDirectAccessor&);
        };

        /**
          * @brief Wrapper for a 2D Linear Bitmap with Direct Access to pixel colors
          */
        template <typename THostType, typename WordType, typename Endianness, bool bitLevelAccess>
        class Linear2DDirectAccessContainer
        {
        public:
            typedef THostType HostType;
            typedef Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess> Iterator;

            /**
              * Constructor.
              * @param buffer bitmap data.
              * @param size size of the bitmap.
              * @param bpp number of bits per pixel of the bitmap.
              * @param width width of the bitmap.
              * @param height height of the bitmap.
              * @param pitch distance between bitmap raws.
              */
            Linear2DDirectAccessContainer(WordType* buffer, UInt32 size, UInt32 bpp, UInt32 width, UInt32 height, UInt32 pitch);

            /**
              * Retrieve an iterator to the first pixel of the bitmap.
              * @return iterator to the first pixel.
              */
            Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess> Begin();

            /**
              * Finalize processing the bitmap.
              */
            void End() const;

            friend class Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess>;
            friend class Linear2DDirectAccessor<HostType, WordType, Endianness, bitLevelAccess>;

        private:
            WordType* m_buffer;
            UInt32 m_size;
            UInt32 m_bpp;
            UInt32 m_width;
            UInt32 m_height;
            UInt32 m_pitch;
            UInt32 m_lineSize;

            enum { c_wordSize = (sizeof(WordType) << 3) - 1 };
        };

        //Template method definitions

        //Iterator
        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess>::Linear2DDirectAccessIterator(Linear2DDirectAccessContainer<HostType, WordType, Endianness, bitLevelAccess>& host, UInt32 bufPos, UInt32 line) :
            m_host(host), m_bufPos(bufPos), m_line(line) {}

        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess> Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess>::operator++ ()
        {
            m_bufPos += m_host.m_bpp;
            if (m_bufPos >= m_host.m_lineSize) {
                m_bufPos = 0;
                m_line ++;
            }
            return *this;
        }

        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess> Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess>::operator++ (Internal::PostIncrementType)
        {
            Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess> prev(*this);
            operator ++();
            return prev;
        }

        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        typename Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess>::Accessor Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess>::operator* ()
        {
            return Accessor(*this);
        }

        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        UInt32 Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess>::GetOffset() const
        {
            if ((m_line > 0) && (m_bufPos == 0)) {
                return (((m_line - 1) * m_host.m_pitch) + m_host.m_lineSize + m_host.c_wordSize) >> 3;
            }
            else {
                return ((m_line * m_host.m_pitch) + m_bufPos + m_host.c_wordSize) >> 3;
            }
        }

        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        bool Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess>::IsAtEnd() const
        {
            return (((m_line * m_host.m_pitch) + m_bufPos) >= (m_host.m_size << 3));
        }

        //Byte accessor
        template <typename HostType, typename WordType, typename Endianness>
        const HostType Linear2DDirectAccessor<HostType, WordType, Endianness, false>::get() const
        {
            UInt32 currentBit = m_source.m_bufPos + (m_source.m_line * m_source.m_host.m_pitch);
            FEATSTD_DEBUG_ASSERT(currentBit <= UInt32(m_source.m_host.m_height * m_source.m_host.m_pitch - m_source.m_host.m_bpp));
            return Internal::Access::GetBitmapBytes<HostType, WordType, Endianness>(m_source.m_host.m_buffer, currentBit);
        }

        template <typename HostType, typename WordType, typename Endianness>
        void Linear2DDirectAccessor<HostType, WordType, Endianness, false>::set(HostType val)
        {
            UInt32 currentBit = m_source.m_bufPos + (m_source.m_line * m_source.m_host.m_pitch);
            FEATSTD_DEBUG_ASSERT(currentBit <= UInt32(m_source.m_host.m_height * m_source.m_host.m_pitch - m_source.m_host.m_bpp));
            Internal::Access::SetBitmapBytes<HostType, WordType, Endianness>(m_source.m_host.m_buffer, currentBit, val);
        }

        template <typename HostType, typename WordType, typename Endianness>
        Linear2DDirectAccessor<HostType, WordType, Endianness, false>::Linear2DDirectAccessor(Linear2DDirectAccessIterator<HostType, WordType, Endianness, false>& source) :
            m_source(source) {}

        //Bit accessor
        template <typename HostType, typename WordType, typename Endianness>
        const HostType Linear2DDirectAccessor<HostType, WordType, Endianness, true>::get() const
        {
            UInt32 currentBit = m_source.m_bufPos + (m_source.m_line * m_source.m_host.m_pitch);
            FEATSTD_DEBUG_ASSERT(currentBit <= UInt32(m_source.m_host.m_height * m_source.m_host.m_pitch - m_source.m_host.m_bpp));
            return Internal::Access::GetBitmapBitsRightToLeft<HostType, WordType, Endianness>(m_source.m_host.m_buffer, currentBit, m_source.m_host.m_bpp);
        }

        template <typename HostType, typename WordType, typename Endianness>
        void Linear2DDirectAccessor<HostType, WordType, Endianness, true>::set(HostType val)
        {
            UInt32 currentBit = m_source.m_bufPos + (m_source.m_line * m_source.m_host.m_pitch);
            FEATSTD_DEBUG_ASSERT(currentBit <= UInt32(m_source.m_host.m_height * m_source.m_host.m_pitch - m_source.m_host.m_bpp));
            Internal::Access::SetBitmapBitsRightToLeft<HostType, WordType, Endianness>(m_source.m_host.m_buffer, currentBit, m_source.m_host.m_bpp, val);
        }

        template <typename HostType, typename WordType, typename Endianness>
        Linear2DDirectAccessor<HostType, WordType, Endianness, true>::Linear2DDirectAccessor(Linear2DDirectAccessIterator<HostType, WordType, Endianness, true>& source) :
            m_source(source) {}

        //Container
        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        Linear2DDirectAccessContainer<HostType, WordType, Endianness, bitLevelAccess>::Linear2DDirectAccessContainer(WordType* buffer, UInt32 size, UInt32 bpp, UInt32 width, UInt32 height, UInt32 pitch) :
            m_buffer(buffer),
            m_size(size),
            m_bpp(bpp),
            m_width(width),
            m_height(height),
            m_pitch(pitch),
            m_lineSize(width * bpp)
        {
            if (m_height == 0) {
                m_size = 0;
            }
            else {
                UInt32 bitmapSize = (((m_height - 1) * m_pitch) + m_lineSize + c_wordSize) >> 3;
                if (m_size > bitmapSize) {
                    m_size = bitmapSize;
                }
            }
        }

        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess> Linear2DDirectAccessContainer<HostType, WordType, Endianness, bitLevelAccess>::Begin()
        {
            return Linear2DDirectAccessIterator<HostType, WordType, Endianness, bitLevelAccess>(*this, 0, 0);
        }
        template <typename HostType, typename WordType, typename Endianness, bool bitLevelAccess>
        void Linear2DDirectAccessContainer<HostType, WordType, Endianness, bitLevelAccess>::End() const
        {
        }
    } //namespace GenericBitmapConvertor
} //namespace Candera

#endif // GENERIC_LINEAR_2D_DIRECT_ACCESSS_BITMAP_H

