//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef GENERIC_BITMAP_PIXEL_CONVERTOR_H
#define GENERIC_BITMAP_PIXEL_CONVERTOR_H

#include <CanderaPlatform/Device/Common/BitmapConverter/GenericHostType.h>

namespace Candera
{
    namespace GenericBitmapConvertor
    {

        /**
         *  @brief Provides a mechanism to convert pixels from one format to another.
         *  Both pixel formats are stored on the same integer type.
         */
        template <typename HostType>
        class GenericBitmapPixelConvertor
        {
        public:
            typedef HostType Type;

            /**
             *  Constructor. This initializer uses the same format as GenericBitmapFormat.
             *  @param dstSize sizes of the destination pixel channel.
             *  @param dstOffset offsets of the destination pixel channel.
             *  @param srcSize offsets of the source pixel channel.
             *  @param srcOffset offsets of the source pixel channel.
             */
            GenericBitmapPixelConvertor(UInt32 dstSize, UInt32 dstOffset, UInt32 srcSize, UInt32 srcOffset, PixelConversionType conversionType);

            /**
             *  Destructor.
             */
            ~GenericBitmapPixelConvertor();

            /**
             *  Conversion operator.
             *  @param src pixel data to convert from.
             *  @return converted pixel data.
             */
            HostType operator() (HostType src) const;

        private:
            GenericBitmapPixelConvertor(const GenericBitmapPixelConvertor<HostType>&);
            const GenericBitmapPixelConvertor<HostType>& operator = (const GenericBitmapPixelConvertor<HostType>&);

            class Convert
            {
            public:
                //select double size for certain operations
                typedef typename HostTypeSelector<(sizeof(HostType) << 4)>::Type HostType2;

                HostType conv(HostType type) const
                {
                    return (this->*m_conv)(type);
                }
                void InitConvert(UInt32 dst, UInt32 src, PixelConversionType type)
                {
                    switch(type) {
                        case RoundLinearConversion:
                    m_dst = (HostType(1) << dst) - 1;
                    m_src = (HostType(1) << src) - 1;
                    m_rem = m_src >> 1;
                    if ((sizeof(HostType) << 3) >= (dst + src + 1)) {
                        m_conv = &Convert::convert;
                    }
                    else {
                        m_conv = &Convert::convert2;
                    }
                            break;
                        case ShiftConversion:
                            if (dst > src) {
                                m_shift = dst - src;
                                m_conv = &Convert::leftShiftConvert;
                }
                            else {
                                m_shift = src - dst;
                                m_conv = &Convert::rightShiftConvert;
                            }

                            break;
                        default:
                            InitIgnore();
                            break;
                    }
                }
                void InitIgnore()
                {
                    m_conv = &Convert::ignore;
                }
            private:
                HostType convert(HostType val) const
                {
                    return (m_dst * val + m_rem) / m_src;
                }
                
                HostType convert2(HostType type) const
                {
                    HostType2 typec = (HostType2)type;
                    HostType2 dstc = (HostType2)m_dst;
                    HostType2 srcc = (HostType2)m_src;
                    HostType2 remc = (HostType2)m_rem;

                    HostType2 res = (dstc * typec + remc) / srcc;

                    return (HostType)res;
                }

                HostType leftShiftConvert(HostType val) const
                {
                    return val << m_shift;
                }

                HostType rightShiftConvert(HostType val) const
                {
                    return val >> m_shift;
                }

                HostType ignore(HostType type) const
                {
                    return type;
                }

                HostType m_dst;
                HostType m_src;
                HostType m_rem;
                UInt32 m_shift;
                HostType (Convert::*m_conv)(HostType type) const;
            };

            class Get
            {
            public:
                HostType get(HostType type) const
                {
                    return (this->*m_get)(type);
                }
                void InitRetrieve(UInt32 pos, UInt32 size)
                {
                    m_shift = pos;
                    m_val = (HostType(1) << size) - 1;
                    m_get = &Get::retrieve;
                }
                void InitDefault(UInt32 size)
                {
                    m_val = (HostType(1) << size) - 1;
                    m_get = &Get::getDefault;
                }
            private:
                HostType retrieve(HostType type) const
                {
                    return (type >> m_shift) & m_val;
                }
                HostType getDefault(HostType /*type*/) const
                {
                    return m_val;
                }

                UInt32 m_shift;
                HostType m_val;
                HostType (Get::*m_get)(HostType type) const;
            };

            class Set
            {
            public:
                void set(HostType& dst, HostType type) const
                {
                    (this->*m_set)(dst, type);
                }
                void InitConjugate(UInt32 pos)
                {
                    m_shift = pos;
                    m_set = &Set::conjugate;
                }
                void InitIgnore()
                {
                    m_set = &Set::ignore;
                }
            private:
                void conjugate(HostType& dst, HostType type) const
                {
                    dst |= type << m_shift;
                }
                void ignore(HostType&, HostType) const
                {
                }

                UInt32 m_shift;
                void (Set::*m_set)(HostType& dst, HostType type) const;
            };

            Convert convertors[4];
            Get getters[4];
            Set setters[4];
        };

        template <typename HostType>
        GenericBitmapPixelConvertor<HostType>::GenericBitmapPixelConvertor(
            UInt32 dstSize,
            UInt32 dstOffset,
            UInt32 srcSize,
            UInt32 srcOffset,
            PixelConversionType conversionType)
        {
            for (UInt i = 0; i < 4; i ++){
                UInt32 ds_ = (dstSize >> (i << 3)) & 0xFFU;
                UInt32 do_ = (dstOffset >> (i << 3)) & 0xFFU;
                UInt32 ss_ = (srcSize >> (i << 3)) & 0xFFU;
                UInt32 so_ = (srcOffset >> (i << 3)) & 0xFFU;

                if (ss_ == 0) {
                    getters[i].InitDefault((i == 0) ? ds_ : 0);
                    ss_ = ds_;
                }
                else {
                    getters[i].InitRetrieve(so_, ss_);
                }

                if (ds_ == 0) {
                    setters[i].InitIgnore();
                }
                else {
                    setters[i].InitConjugate(do_);
                }

                if (ds_ == ss_) {
                    convertors[i].InitIgnore();
                }
                else {
                    convertors[i].InitConvert(ds_, ss_, conversionType);
                }
            }
        }

        template <typename HostType>
        GenericBitmapPixelConvertor<HostType>::~GenericBitmapPixelConvertor()
        {
        }

        template <typename HostType>
        HostType GenericBitmapPixelConvertor<HostType>::operator()(HostType src) const
        {
            HostType out = HostType();
            for (Int i = 0; i < 4; i ++){
                setters[i].set(out, convertors[i].conv(getters[i].get(src)));
            }
            return out;
        }

    } //namespace GenericBitmapConvertor

} //namespace Candera

#endif //GENERIC_BITMAP_PIXEL_CONVERTOR_H

