//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef GENERIC_BITMAP_EXTENSION_H
#define GENERIC_BITMAP_EXTENSION_H

#include <Candera/Environment.h>
#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapAccess.h>

namespace Candera
{
    /**
     *  @brief Provides format description for bitmap data. This type of format is used by
     *  the software bitmap converter and by a few platforms.
     */
    class GenericBitmapFormat
    {
    public:
        /**
          * Create a R8G8B8A8 format descriptor.
          */
        GenericBitmapFormat() : 
            m_access(), m_endianness(),
            m_bitsPerPixel(32), m_offset(0x18100800), m_size(0x08080808) {}

        /**
          * Create a format descriptor as defined by the input parameters.
          * Each pixel is described as an RGBA color. Packed as described by
          * see off and size. Information for each channel is stored on 8 bits,
          * in the specified order (that is: the most significant 8 bits are used
          * to describe the Red channel, the following 8 to describe the Green channel, and so on.)
          * @param bpp number of bits per pixel of the given format.
          * @param off describes the position of the channel data starting with the least significant bit.
          * @param size describes the number of bits used for each channel.
          * @param access generic bitmap access
          */
        GenericBitmapFormat(UInt8 bpp, UInt32 off, UInt32 size, 
            GenericBitmapAccess access = GenericBitmapAccess(),
            GenericBitmapEndianness endianness = GenericBitmapEndianness()) : 
            m_access(access), m_endianness(endianness), 
            m_bitsPerPixel(bpp), m_offset(off), m_size(size) {}

        GenericBitmapFormat(const GenericBitmapFormat* src):
            m_access(src->m_access), m_endianness(src->m_endianness),
            m_bitsPerPixel(src->m_bitsPerPixel), m_offset(src->m_offset), m_size(src->m_size)
        {}

        /**
          * Get the number of bits per pixel.
          * @return number of bits per pixel.
          */
        UInt8 GetBitsPerPixel() const { return m_bitsPerPixel; }
        /**
          * Set the number of bits per pixel.
          * @param val number of bits per pixel.
          */
        void SetBitsPerPixel(UInt8 val) { m_bitsPerPixel = val; }


        /**
          * Get the offsets for all channels.
          * @return offsets for all channels.
          */
        UInt32 GetOffset() const { return m_offset; }
        /**
          * Set the offsets for all channels.
          * @param val offsets for all channels.
          */
        void SetOffset(UInt32 val) { m_offset = val; }

        /**
          * Get the offset for the red channel.
          * @return offset for the red channel.
          */
        UInt8 GetRedOffset() const { return Get(m_offset, redShift); }
        /**
          * Set the offset for the red channel.
          * @param val offset for the red channel.
          */
        void SetRedOffset(UInt8 val) { Set(m_offset, val, redShift); }

        /**
          * Get the offset for the green channel.
          * @return offset for the green channel.
          */
        UInt8 GetGreenOffset() const { return Get(m_offset, greenShift); }
        /**
          * Set the offset for the green channel.
          * @param val offset for the green channel.
          */
        void SetGreenOffset(UInt8 val) { Set(m_offset, val, greenShift); }

        /**
          * Get the offset for the blue channel.
          * @return offset for the blue channel.
          */
        UInt8 GetBlueOffset() const { return Get(m_offset, blueShift); }
        /**
          * Set the offset for the blue channel.
          * @param val offset for the blue channel.
          */
        void SetBlueOffset(UInt8 val) { Set(m_offset, val, blueShift); }

        /**
          * Get the offset for the alpha channel.
          * @return offset for the alpha channel.
          */
        UInt8 GetAlphaOffset() const { return Get(m_offset, alphaShift); }
        /**
          * Set the offset for the alpha channel.
          * @param val offset for the alpha channel.
          */
        void SetAlphaOffset(UInt8 val) { Set(m_offset, val, alphaShift); }



        /**
          * Get the sizes for all channels.
          * @return sizes for all channels.
          */
        UInt32 GetSize() const { return m_size; }
        /**
          * Set the sizes for all channels.
          * @param val sizes for all channels.
          */
        void SetSize(UInt32 val) { m_size = val; }

        /**
          * Get the size for the red channel.
          * @return size for the red channel.
          */
        UInt8 GetRedSize() const { return Get(m_size, redShift); }
        /**
          * Set the size for the red channel.
          * @param val size for the red channel.
          */
        void SetRedSize(UInt8 val) { Set(m_size, val, redShift); }

        /**
          * Get the size for the green channel.
          * @return size for the green channel.
          */
        UInt8 GetGreenSize() const { return Get(m_size, greenShift); }
        /**
          * Set the size for the green channel.
          * @param val size for the green channel.
          */
        void SetGreenSize(UInt8 val) { Set(m_size, val, greenShift); }

        /**
          * Get the size for the blue channel.
          * @return size for the blue channel.
          */
        UInt8 GetBlueSize() const { return Get(m_size, blueShift); }
        /**
          * Set the size for the blue channel.
          * @param val size for the blue channel.
          */
        void SetBlueSize(UInt8 val) { Set(m_size, val, blueShift); }

        /**
          * Get the size for the alpha channel.
          * @return size for the alpha channel.
          */
        UInt8 GetAlphaSize() const { return Get(m_size, alphaShift); }
        /**
          * Set the size for the alpha channel.
          * @param val size for the alpha channel.
          */
        void SetAlphaSize(UInt8 val) { Set(m_size, val, alphaShift); }


        /**
          * Get pixel access mode. This determines how pixels are layed out 
          * within the bitmap.
          * @return size for the alpha channel.
          */
        GenericBitmapAccess GetBitmapAccess() const { return m_access; }
        /**
          * Set pixel access mode.
          * @param val size for the alpha channel.
          */
        void SetBitmapAccess(GenericBitmapAccess val) { m_access = val; }

        /**
          * Get word endianness. This determines how bytes are layed out within
          * the pixel data.
          * @return size for the alpha channel.
          */
        GenericBitmapEndianness GetBitmapEndianness() const { return m_endianness; }
        /**
          * Set word endianness.
          * @param val size for the alpha channel.
          */
        void SetBitmapEndianness(GenericBitmapEndianness val) { m_endianness = val; }

    private:
        enum {
            redShift = 24,      ///< red shift, 24 bits.
            greenShift = 16,    ///< green shift, 16 bits.
            blueShift = 8,      ///< blue shift, 8 bits.
            alphaShift = 0      ///< alpha shift, 0 bits.
        };

        GenericBitmapAccess m_access;
        GenericBitmapEndianness m_endianness;

        UInt8 m_bitsPerPixel;

        UInt32 m_offset;
        UInt32 m_size;

        static inline UInt8 Get(UInt32 src, UInt32 pos) { return UInt8(src >> pos); }
        static inline void Set(UInt32& dst, UInt8 val, UInt32 pos) { dst = (dst & ~(0xffUL << pos)) | (UInt32(val) << pos); }

    }; //class GenericBitmapFormat

} //namespace Candera

#endif //GENERIC_BITMAP_EXTENSION_H

