//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GENERIC_BITMAP_CONVERTOR_IMPL_H)
#define GENERIC_BITMAP_CONVERTOR_IMPL_H

#include "GenericBitmapConvertor.h"

#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapBaseConvertor.h>
#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapExtendedConvertor.h>

#include <CanderaPlatform/Device/Common/BitmapConverter/GenericLinear2DDirectAccessBitmap.h>
#include <CanderaPlatform/Device/Common/BitmapConverter/GenericRunLengthEncodedBitmap.h>
#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapPixelConvertor.h>

#include <CanderaPlatform/Device/Common/BitmapConverter/GenericHostType.h>

namespace Candera
{
    namespace GenericBitmapConvertor
    {


class BaseConvertor
{
public:
    virtual bool Convert(
        const ConvertorInfo& dst,
        UInt8* dstBuffer, UInt32* dstBufferSize,
        const ConvertorInfo& src,
        const UInt8* srcBuffer, UInt32 srcBufferSize,
        PixelConversionType conversionType) const = 0;
};

template <class DstContainer, class SrcContainer, class PixelConvertor>
class Convertor : public BaseConvertor
{
public:
    virtual bool Convert(
        const ConvertorInfo& dst,
        UInt8* dstBuffer, UInt32* dstBufferSize,
        const ConvertorInfo& src,
        const UInt8* srcBuffer, UInt32 srcBufferSize,
        PixelConversionType conversionType) const
    {
        DstContainer dstContainer(dst, dstBuffer, *dstBufferSize);
        SrcContainer srcContainer(src, srcBuffer, srcBufferSize);

        DstIterator dstIt = dstContainer.m_container.Begin();
        SrcIterator srcIt = srcContainer.m_container.Begin();

        PixelConvertor convertor(dst.size, dst.offset, src.size, src.offset, conversionType);
        while ((!dstIt.IsAtEnd()) && (!srcIt.IsAtEnd())) {
            (*dstIt++).set(static_cast<DstHostType>(convertor(static_cast<ConvHostType>((*srcIt++).get()))));
        }

        dstContainer.m_container.End();
        srcContainer.m_container.End();

        *dstBufferSize = dstIt.GetOffset();

        return srcIt.IsAtEnd();
    }

    static const BaseConvertor* Get()
    {
        static Convertor convertor;
        return &convertor;
    }

protected:
    Convertor() {}

private:
    typedef typename DstContainer::Base::Iterator DstIterator;
    typedef typename SrcContainer::Base::Iterator SrcIterator;

    typedef typename DstContainer::Base::HostType DstHostType;
    typedef typename SrcContainer::Base::HostType SrcHostType;

    typedef typename PixelConvertor::Type ConvHostType;
};

Int MapAccessToAccessContainer(GenericBitmapAccess access)
{
    Int ret;
    switch(access) {
    case Linear2DDirectAccess:
        ret = 0; break;
    case RunLengthEncoding:
    case OneBasedRunLengthEncoding:
        ret = 1; break;
    default:
        ret = -1; break;
    }
    return ret;
}

UInt32 MapAccessToEncodingBase(GenericBitmapAccess access)
{
    Int ret;
    switch(access) {
    case RunLengthEncoding:
        ret = 0; break;
    case OneBasedRunLengthEncoding:
        ret = 1; break;
    default:
        ret = 0; break;
    }
    return ret;
}

typedef UInt8 LinearBufferHostType;

template<typename HostType, typename EndiannessChoice, bool destination, bool hostLarger>
class AccessContainerLinear2D
{
public:
    AccessContainerLinear2D(const ConvertorInfo& info, void* buffer, UInt32 size) :
        m_container(
                FeatStd::Internal::PointerToPointer<LinearBufferHostType*>(buffer),
                size,
                info.bpp,
                info.width,
                info.height,
                info.pitch)
    {}
    AccessContainerLinear2D(const ConvertorInfo& info, const void* buffer, UInt32 size) :
        FEATSTD_LINT_NEXT_EXPRESSION(925, "Violates MISRA C++ 2008 Required Rule 5-2-8: const cast")
        m_container(
                FeatStd::Internal::PointerToPointer<LinearBufferHostType*>(const_cast<void*>(buffer)),
                size,
                info.bpp,
                info.width,
                info.height,
                info.pitch)
    {}
    typedef Linear2DDirectAccessContainer<
        HostType,
        LinearBufferHostType,
        typename EndiannessChoice::Word,
        hostLarger> Base;
    Base m_container;
};

typedef UInt32 RLEBufferHostType;

template<typename HostType, typename EndiannessChoice>
class AccessContainerRLE
{
public:
    AccessContainerRLE(const ConvertorInfo& info, void* buffer, UInt32 size) :
        m_container(
        FeatStd::Internal::PointerToPointer<RLEBufferHostType*>(buffer),
        size,
        info.bpp,
        MapAccessToEncodingBase(info.access))
    {
    }

    typedef RunLengthEncodingContainer<
        HostType, RLEBufferHostType,
        typename EndiannessChoice::Word,
        typename EndiannessChoice::Pixel> Base;
    Base m_container;
};

template<typename HostType, typename EndiannessChoice>
class AccessContainerRLD
{
public:
    AccessContainerRLD(const ConvertorInfo& info, const void* buffer, UInt32 size) :
        m_container(
        FeatStd::Internal::PointerToPointer<const RLEBufferHostType*>(buffer),
        size,
        info.bpp,
        MapAccessToEncodingBase(info.access))
    {
    }
    typedef RunLengthDecodingContainer<
        HostType, RLEBufferHostType,
        typename EndiannessChoice::Word,
        typename EndiannessChoice::Pixel> Base;
    Base m_container;
};

template<class HostSelector, Int dstSize, Int srcSize>
class SelectorGetter {};

template<Int start, Int end>
class SelectorSpliter
{
public:
    static const Int c_mid = (start + end) / 2;
    static const Int c_nxMid = c_mid + 1;
};
template<Int val>
class SelectorSpliter<val, val>
{
public:
    static const Int c_mid = val;
    static const Int c_nxMid = val;
};

template<class HostSelector, Int dstStart, Int dstEnd, Int srcStart, Int srcEnd>
class Selector
{
public:
    typedef typename HostSelector::ReturnType ReturnType;
    typedef typename HostSelector::InputType InputType;

    static ReturnType Get(InputType dst, InputType src)
    {
        Int dstVal = HostSelector::GetVal(dst);
        Int srcVal = HostSelector::GetVal(src);

        const Int dstMid = SelectorSpliter<dstStart, dstEnd>::c_mid;
        const Int dstNxMid = SelectorSpliter<dstStart, dstEnd>::c_nxMid;
        const Int srcMid = SelectorSpliter<srcStart, srcEnd>::c_mid;
        const Int srcNxMid = SelectorSpliter<srcStart, srcEnd>::c_nxMid;

        if (dstVal > dstMid) {
            if (srcVal > srcMid) {
                return Selector<HostSelector, dstNxMid, dstEnd, srcNxMid, srcEnd>::Get(dst, src);
            }
            else {
                return Selector<HostSelector, dstNxMid, dstEnd, srcStart, srcMid>::Get(dst, src);
            }
        }
        else {
            if (srcVal > srcMid) {
                return Selector<HostSelector, dstStart, dstMid, srcNxMid, srcEnd>::Get(dst, src);
            }
            else {
                return Selector<HostSelector, dstStart, dstMid, srcStart, srcMid>::Get(dst, src);
            }
        }
    }
};

template<class HostSelector, Int dstFound, Int srcFound>
class Selector<HostSelector, dstFound, dstFound, srcFound, srcFound>
{
public:
    typedef typename HostSelector::ReturnType ReturnType;
    typedef typename HostSelector::InputType InputType;

    static ReturnType Get(InputType dst, InputType src)
    {
        Int dstVal = HostSelector::GetVal(dst);
        Int srcVal = HostSelector::GetVal(src);

        if ((dstVal == dstFound) && (srcVal == srcFound)) {
            return SelectorGetter<HostSelector, dstFound, srcFound>::Get(dst, src);
        }

        return 0;
    }
};

template <typename Endianness> class EndiannessDual {};
FEATSTD_LINT_SYMBOL(753, Candera::GenericBitmapConvertor::EndiannessDual<Candera::MemoryManagement::Internal::BigEndian>,"Violates MISRA C++ 2008 Required Rule 0-1-5: template provided by intention, even if unused")
template <> class EndiannessDual<BigEndian> { public: typedef LittleEndian Type; };
template <> class EndiannessDual<LittleEndian> { public: typedef BigEndian Type; };

template <Int endiannessIdx> class EndiannessMap {};
template <> class EndiannessMap<LittleEndianBitmapWord>
{
public:
    struct Type {
        typedef LittleEndian Word;
        typedef HostEndianness Pixel;
    };
};
template <> class EndiannessMap<BigEndianBitmapWord>
{
public:
    struct Type {
        typedef BigEndian Word;
        typedef HostEndianness Pixel;
    };
};
template <> class EndiannessMap<BitEndianBitmapWordLittleEndianBitmapPixel>
{
public:
    struct Type {
        typedef BigEndian Word;
        typedef EndiannessDual<HostEndianness>::Type Pixel;
    };
};

template <UInt32 dstSize, UInt32 srcSize>
class EndiannessSelector
{
public:
    typedef const BaseConvertor* ReturnType;
    typedef const ConvertorInfo& InputType;

    static Int GetVal(InputType obj)
    {
        return Int(obj.endianness);
    }
};

template <typename DstEndian, typename SrcEndian, UInt32 dstSize, UInt32 srcSize>
class AccessSelector
{
public:
    typedef const BaseConvertor* ReturnType;
    typedef const ConvertorInfo& InputType;

    static Int GetVal(InputType obj)
    {
        return MapAccessToAccessContainer(obj.access);
    }
};

template<UInt32 dstSize, UInt32 srcSize, Int dstEndian, Int srcEndian>
class SelectorGetter<EndiannessSelector<dstSize, srcSize>, dstEndian, srcEndian>
{
public:
    typedef typename EndiannessSelector<dstSize, srcSize>::ReturnType ReturnType;
    typedef typename EndiannessSelector<dstSize, srcSize>::InputType InputType;

    static ReturnType Get(InputType dst, InputType src)
    {
        typedef typename EndiannessMap<dstEndian>::Type DstEndianness;
        typedef typename EndiannessMap<srcEndian>::Type SrcEndianness;

        typedef Selector<
            AccessSelector<DstEndianness, SrcEndianness, dstSize, srcSize>,
            0, 1, 0, 1>
            ConvertorSelector;
        return ConvertorSelector::Get(dst, src);
    }
};

#define LINEAR2D_TO_LINEAR2D 0, 0
#define RLD_TO_RLE 1, 1
#define LINEAR2D_TO_RLE 1, 0
#define RLD_TO_LINEAR2D 0, 1

template<typename DstEndian, typename SrcEndian, UInt32 dstSize, UInt32 srcSize, Int dstAccess, Int srcAccess>
class SelectorGetter<AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>, dstAccess, srcAccess>
{
public:
    typedef typename AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>::ReturnType ReturnType;
    typedef typename AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>::InputType InputType;

    static ReturnType Get(InputType /*dst*/, InputType /*src*/);
};

template<UInt32 dstSize, UInt32 srcSize>
class ConvertorTypes
{
public:
        typedef typename HostTypeSelector<dstSize>::Type DstHostType;
        typedef typename HostTypeSelector<srcSize>::Type SrcHostType;
        typedef typename HostTypeSelector<((dstSize > srcSize) ? dstSize : srcSize)>::Type PixelType;
        typedef GenericBitmapPixelConvertor<PixelType> PixelConvertor;
};

template<typename DstEndian, typename SrcEndian, UInt32 dstSize, UInt32 srcSize>
class SelectorGetter<AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>, RLD_TO_RLE>
{
public:
    typedef typename AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>::ReturnType ReturnType;
    typedef typename AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>::InputType InputType;

    static ReturnType Get(InputType /*dst*/, InputType /*src*/)
    {
        FEATSTD_LINT_CURRENT_SCOPE(948, "Violates MISRA C++ 2008 Required Rule 0-1-9: depending on template type")
        FEATSTD_LINT_CURRENT_SCOPE(944, "Violates MISRA C++ 2008 Required Rule 0-1-1: depending on template type")
        typedef AccessContainerRLE<typename ConvertorTypes<dstSize, srcSize>::DstHostType, DstEndian> DstContainer;
        typedef AccessContainerRLD<typename ConvertorTypes<dstSize, srcSize>::SrcHostType, SrcEndian> SrcContainer;

        return Convertor<DstContainer, SrcContainer, typename ConvertorTypes<dstSize, srcSize>::PixelConvertor>::Get();
    }
};

template<typename DstEndian, typename SrcEndian, UInt32 dstSize, UInt32 srcSize>
class SelectorGetter<AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>, LINEAR2D_TO_RLE>
{
public:
    typedef typename AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>::ReturnType ReturnType;
    typedef typename AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>::InputType InputType;

    static ReturnType Get(InputType /*dst*/, InputType /*src*/)
    {
        FEATSTD_LINT_CURRENT_SCOPE(948, "Violates MISRA C++ 2008 Required Rule 0-1-9: depending on template type")
        FEATSTD_LINT_CURRENT_SCOPE(944, "Violates MISRA C++ 2008 Required Rule 0-1-1: depending on template type")
        typedef AccessContainerRLE<typename ConvertorTypes<dstSize, srcSize>::DstHostType, DstEndian> DstContainer;
                static const bool isLarge = (sizeof(typename ConvertorTypes<dstSize, srcSize>::SrcHostType) << 3) > srcSize;
        typedef AccessContainerLinear2D<typename ConvertorTypes<dstSize, srcSize>::SrcHostType, SrcEndian, isLarge, false> SrcContainer;

        return Convertor<DstContainer, SrcContainer, typename ConvertorTypes<dstSize, srcSize>::PixelConvertor>::Get();
    }
};

template<typename DstEndian, typename SrcEndian, UInt32 dstSize, UInt32 srcSize>
class SelectorGetter<AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>, RLD_TO_LINEAR2D>
{
public:
    typedef typename AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>::ReturnType ReturnType;
    typedef typename AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>::InputType InputType;

    static ReturnType Get(InputType /*dst*/, InputType /*src*/)
    {
        FEATSTD_LINT_CURRENT_SCOPE(948, "Violates MISRA C++ 2008 Required Rule 0-1-9: depending on template type")
        FEATSTD_LINT_CURRENT_SCOPE(944, "Violates MISRA C++ 2008 Required Rule 0-1-1: depending on template type")
                static const bool isLarge = (sizeof(typename ConvertorTypes<dstSize, srcSize>::DstHostType) << 3) > dstSize;
        typedef AccessContainerLinear2D<typename ConvertorTypes<dstSize, srcSize>::DstHostType, DstEndian, true, isLarge> DstContainer;
        typedef AccessContainerRLD<typename ConvertorTypes<dstSize, srcSize>::SrcHostType, SrcEndian> SrcContainer;

        return Convertor<DstContainer, SrcContainer, typename ConvertorTypes<dstSize, srcSize>::PixelConvertor>::Get();
    }
};

template<typename DstEndian, typename SrcEndian, UInt32 dstSize, UInt32 srcSize>
class SelectorGetter<AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>, LINEAR2D_TO_LINEAR2D>
{
public:
    typedef typename AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>::ReturnType ReturnType;
    typedef typename AccessSelector<DstEndian, SrcEndian, dstSize, srcSize>::InputType InputType;

    static ReturnType Get(InputType /*dst*/, InputType /*src*/)
    {
        FEATSTD_LINT_CURRENT_SCOPE(948, "Violates MISRA C++ 2008 Required Rule 0-1-9: depending on template type")
        FEATSTD_LINT_CURRENT_SCOPE(944, "Violates MISRA C++ 2008 Required Rule 0-1-1: depending on template type")
                static const bool dstIsLarge = (sizeof(typename ConvertorTypes<dstSize, srcSize>::DstHostType) << 3) > dstSize;
        typedef AccessContainerLinear2D<typename ConvertorTypes<dstSize, srcSize>::DstHostType, DstEndian, true, dstIsLarge> DstContainer;
                static const bool srcIsLarge = (sizeof(typename ConvertorTypes<dstSize, srcSize>::SrcHostType) << 3) > srcSize;
        typedef AccessContainerLinear2D<typename ConvertorTypes<dstSize, srcSize>::SrcHostType, SrcEndian, false, srcIsLarge> SrcContainer;

        return Convertor<DstContainer, SrcContainer, typename ConvertorTypes<dstSize, srcSize>::PixelConvertor>::Get();
    }
};


template<Int index> class SelectBpp { public: static const UInt32 c_elem = ~static_cast<UInt32>(0); };
template <> class SelectBpp<0> { public: static const UInt32 c_elem = 1; };
template <> class SelectBpp<1> { public: static const UInt32 c_elem = 2; };
template <> class SelectBpp<2> { public: static const UInt32 c_elem = 4; };
template <> class SelectBpp<3> { public: static const UInt32 c_elem = 8; };
template <> class SelectBpp<4> { public: static const UInt32 c_elem = 16; };
template <> class SelectBpp<5> { public: static const UInt32 c_elem = 24; };
template <> class SelectBpp<6> { public: static const UInt32 c_elem = 32; };

FEATSTD_LINT_NEXT_EXPRESSION(948, "Violates MISRA C++ 2008 Required Rule 0-1-9: depending on template type")
template <Int index = 0, bool available = (SelectBpp<index>::c_elem != ~static_cast<UInt32>(0))>
class FindMaxElem {
public:
    static const Int c_index = index;
};
template <Int index>
class FindMaxElem<index, true> {
public:
    static const Int c_index = FindMaxElem<index + 1>::c_index;
};

class BppMapper
{
public:
    static Int GetIndex(UInt32 bpp) { return GetBppToIndex().GetBppIndex(bpp); }
    static bool IsValid(UInt32 bpp) { return GetBppToIndex().IsBppValid(bpp); }

    static const UInt32 g_size = FindMaxElem<>::c_index;
private:
    Int GetBppIndex(UInt32 bpp) const
    {
        if (IsBppValid(bpp)) {
            return m_invertedMap[bpp];
        }
        return 0;
    }

    bool IsBppValid(UInt32 bpp) const
    {
        return
            (bpp < (sizeof(m_invertedMap) / sizeof(m_invertedMap[0]))) &&
            (m_invertedMap[bpp] >= 0);
    }

    static const BppMapper& GetBppToIndex()
    {
        static const UInt32 supportedBitsPerPixel[] = {
            SelectBpp<0>::c_elem,
            SelectBpp<1>::c_elem,
            SelectBpp<2>::c_elem,
            SelectBpp<3>::c_elem,
            SelectBpp<4>::c_elem,
            SelectBpp<5>::c_elem,
            SelectBpp<6>::c_elem
        };
        static const UInt32 supportedMapSize = sizeof(supportedBitsPerPixel) / sizeof(supportedBitsPerPixel[0]);
        static const BppMapper s_bppMapper(supportedMapSize, supportedBitsPerPixel);
        return s_bppMapper;
    }

    BppMapper(const UInt32 supportedMapSize, const UInt32 supportedBitsPerPixel[])
    {
        const UInt32 invertedMapSize = sizeof(m_invertedMap) / sizeof(m_invertedMap[0]);
        for (UInt32 i = 0; i < invertedMapSize; i++){
            m_invertedMap[i] = -1;
        }

        for (UInt32 i = 0; i < supportedMapSize; i++){
            m_invertedMap[supportedBitsPerPixel[i]] = Int(i);
        }
    }

    static const UInt32 c_invertedMapSize = SelectBpp<BppMapper::g_size - 1>::c_elem + 1;
    Int m_invertedMap[c_invertedMapSize];
};


class BppSelector
{
public:
    static const Int c_maxBits = static_cast<Int>(BppMapper::g_size) - 1;

    typedef const BaseConvertor* ReturnType;
    typedef const ConvertorInfo& InputType;

    static Int GetVal(InputType obj)
    {
        return BppMapper::GetIndex(obj.bpp);
    }
};


template<Int dstSize, Int srcSize>
class SelectorGetter<BppSelector, dstSize, srcSize>
{
public:
    typedef BppSelector::ReturnType ReturnType;
    typedef BppSelector::InputType InputType;

    static ReturnType Get(InputType dst, InputType src)
    {
        if ((!BppMapper::IsValid(dst.bpp)) || (!BppMapper::IsValid(src.bpp)))
        {
            return 0;
        }

        const UInt32 dstBpp = SelectBpp<dstSize>::c_elem;
        const UInt32 srcBpp = SelectBpp<srcSize>::c_elem;

        typedef Selector<
            EndiannessSelector<dstBpp, srcBpp>,
            0,
            Int(GenericBitmapEndiannessItemCount) - 1,
            0,
            Int(GenericBitmapEndiannessItemCount) - 1>
            ConvertorSelector;
        return ConvertorSelector::Get(dst, src);
    }
};


}}
#endif
