//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GENERIC_BITMAP_CONVERTOR_H)
#define GENERIC_BITMAP_CONVERTOR_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/Bitmap.h>

#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapAccess.h>

namespace Candera
{
    class GenericBitmapFormat;

    namespace GenericBitmapConvertor
    {
        /**
         *  @brief This information is used internally by the converter to
         *  iterate through the bitmap and convert its pixels.
         */
        struct ConvertorInfo {
            UInt32 width;   ///<width of the bitmap
            UInt32 height;  ///<height of the bitmap
            UInt32 pitch;   ///<distance between the raws of the bitmap.
            UInt32 bpp;     ///<number of bits per bitmap pixel.
            UInt32 size;    ///<number of bits per channel of bitmap pixel.
            UInt32 offset;  ///<position of channel within a bitmap pixel.

            GenericBitmapAccess access;  ///<describes the access type of the bitmap.
            GenericBitmapEndianness endianness; ///<describes the endianess of the bitmap words.
        };

        /**
         * @brief Type of pixel conversion.
         *
         * This defines the conversion applied to each channel.
         * - srcBpp is the bit count of the source format.
         * - dstBpp is the bit count of the destination format.
         * - srcMax is the maximum value that the source format can take. ( (1 << srcBpp) - 1 )
         * - dstMax is the maximum value that the destination format can take. ( (1 << dstBpp) - 1 )
         * - val is the value of the current channel being transformed.
         */
        enum PixelConversionType
        {
            RoundLinearConversion, ///< round(val * (dstMax / srcMax)).
            ShiftConversion ///< floor(val * pow(2, (dstBpp - srcBpp)).
        };

        /**
         *  Convert one bitmap format to another.
         *  Only if the extended properties for one of the bitmaps are
         *  missing, the regular pixel properties of the bitmap come into play.
         *
         *  Each channel from the source format is transformed into the same
         *  channel in the destination format. When channels overlap, the 
         *  information of previous channels is overwritten with information
         *  from subsequent channels.
         *
         *  @see GetGenericBitmapBaseConvertorInfo and @see GetGenericBitmapExtendedConvertorInfo
         *  @param dstWidth Width of the destination bitmap.
         *  @param dstHeight Height of the destination bitmap.
         *  @param dstPixelFormat Pixel format of the destination bitmap.
         *  @param dstColorPackAllignment Raw alignment of the destination bitmap
         *  @param dstExtProp Extended pixel properties of the destination bitmap
         *  @param srcWidth Width of the source bitmap.
         *  @param srcHeight Height of the source bitmap.
         *  @param srcPixelFormat Pixel format of the source bitmap.
         *  @param srcColorPackAllignment Raw alignment of the source bitmap
         *  @param srcExtProp Extended pixel properties of the source bitmap
         *  @param conversionType Type of pixel conversion.
         *  @return true if conversion succeeds.
         */
        bool Convert(
            UInt32 dstWidth, UInt32 dstHeight,
            Int dstPixelFormat,
            Bitmap::PackAlignment dstColorPackAllignment,
            UInt8* dstBuffer, UInt32* dstBufferSize,
            const GenericBitmapFormat* dstExtProp,
            UInt32 srcWidth, UInt32 srcHeight,
            Int srcPixelFormat, 
            Bitmap::PackAlignment srcColorPackAllignment,
            const UInt8* srcBuffer, UInt32 srcBufferSize,
            const GenericBitmapFormat* srcExtProp,
            PixelConversionType conversionType);

        bool Convert(
            UInt32 dstWidth, UInt32 dstHeight,
            Int dstPixelFormat,
            Bitmap::PackAlignment dstColorPackAllignment,
            UInt8* dstBuffer, UInt32* dstBufferSize,
            const GenericBitmapFormat* dstExtProp,
            UInt32 srcWidth, UInt32 srcHeight,
            Int srcPixelFormat,
            Bitmap::PackAlignment srcColorPackAllignment,
            const UInt8* srcBuffer, UInt32 srcBufferSize,
            const GenericBitmapFormat* srcExtProp);
    } //namespace GenericBitmapConvertor

} //namespace Candera

#endif // GENERIC_BITMAP_CONVERTOR_H

