//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef GENERIC_BITMAP_ACCESS_INTERNAL_H
#define GENERIC_BITMAP_ACCESS_INTERNAL_H

#include <Candera/Environment.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>
#include <Candera/System/MemoryManagement/Endianness.h>

namespace Candera
{
    namespace GenericBitmapConvertor
    {

using MemoryManagement::LittleEndian;
using MemoryManagement::BigEndian;
using MemoryManagement::HostEndianness;

        namespace Internal {

typedef int PostIncrementType;

class Access {
public:
    template<typename PixelType, typename WordType, typename Endianness>
    static PixelType GetBitmapBytes(const WordType* data, UInt32 bitPos);

    template<typename PixelType, typename WordType, typename Endianness>
    static void SetBitmapBytes(WordType* data, UInt32 bitPos, PixelType val);

    template<typename PixelType, typename WordType, typename Endianness>
    static PixelType GetBitmapBits(const WordType* data, UInt32 bitPos, UInt32 bitNum);

    template<typename PixelType, typename WordType, typename Endianness>
    static void SetBitmapBits(WordType* data, UInt32 bitPos, UInt32 bitNum, PixelType val);

    template<typename PixelType, typename WordType, typename Endianness>
    static PixelType GetBitmapBitsRightToLeft(const WordType* data, UInt32 bitPos, UInt32 bitNum);

    template<typename PixelType, typename WordType, typename Endianness>
    static void SetBitmapBitsRightToLeft(WordType* data, UInt32 bitPos, UInt32 bitNum, PixelType val);

private:
    //helper functions
    template <typename T0, typename T1, bool TMax = (sizeof(T0) < sizeof(T1))>
    class WordTypeMaximizer{
    public:
        typedef T0 Type;
    };

    template <typename T0, typename T1>
    class WordTypeMaximizer<T0, T1, true> {
    public:
        typedef T1 Type;
    };

    template<typename WordType>
    static inline UInt32 GetBitCount();

    template<typename WordType>
    static inline WordType GetBitMask(UInt32 bitnum);
};

template <typename Src, typename Dst>
struct SameClass {
    static const Int value = 0;
};
template <typename Same>
struct SameClass<Same, Same> {
    static const Int value = 1;
};

template<typename WordType, typename Endianness, 
    bool mustConvert = (sizeof(WordType) > 1) &&
        (SameClass<Endianness, HostEndianness>::value == 0)>
struct EndiannessConverter {
    // mustConvert == true;
    static WordType ToHost(WordType item)
    {
        return MemoryManagement::Internal::Converter<
            Endianness, HostEndianness, WordType>::Convert(item); 
    }
    static WordType FromHost(WordType item)
    {
        return MemoryManagement::Internal::Converter<
            HostEndianness, Endianness, WordType>::Convert(item); 
    }
    static void CopyToHost(void* dst, const WordType* src, UInt32 size)
    {
        WordType *typedDst = static_cast<const WordType*>(dst);
        while (size-- > 0) {
            *typedDst++ = ToHost(*src++);
        }
    }
    static void CopyFromHost(WordType* dst, const void* src, UInt32 size)
    {
        WordType *typedSrc = src;
        while (size-- > 0) {
            *dst++ = FromHost(*typedSrc++);
        }
    }
};

template<typename WordType, typename Endianness>
struct EndiannessConverter<WordType, Endianness, false> {
    // mustConvert == false;
    static WordType ToHost(WordType item) { return item; }
    static WordType FromHost(WordType item) { return item; }
    static void CopyToHost(void* dst, const WordType* src, UInt32 size)
    {
        MemoryPlatform::Copy(dst, src, size * sizeof(WordType));
    }
    static void CopyFromHost(WordType* dst, const void* src, UInt32 size)
    {
        MemoryPlatform::Copy(dst, src, size * sizeof(WordType));
    }
};

//byte accessor
template<typename PixelType, typename WordType, typename Endianness>
PixelType Access::GetBitmapBytes(const WordType* data, UInt32 bitPos)
{
    FEATSTD_COMPILETIME_ASSERT((sizeof(PixelType) % sizeof(WordType)) == 0);

    const UInt32 wordBits = GetBitCount<WordType>();
    FEATSTD_DEBUG_ASSERT((bitPos % wordBits) == 0);
    data += bitPos / wordBits;

    PixelType val;
    EndiannessConverter<WordType, Endianness>::CopyToHost(
        &val, data, (sizeof(PixelType) / sizeof(WordType)));

    return val;
}

template<typename PixelType, typename WordType, typename Endianness>
void Access::SetBitmapBytes(WordType* data, UInt32 bitPos, PixelType val)
{
    FEATSTD_COMPILETIME_ASSERT((sizeof(PixelType) % sizeof(WordType)) == 0);

    const UInt32 wordBits = GetBitCount<WordType>();
    FEATSTD_DEBUG_ASSERT((bitPos % wordBits) == 0);
    data += bitPos / wordBits;
                
    EndiannessConverter<WordType, Endianness>::CopyFromHost(
        data, &val, (sizeof(PixelType) / sizeof(WordType)));
}

//bit accessor
template<typename PixelType, typename WordType, typename Endianness>
PixelType Access::GetBitmapBits(const WordType* data, UInt32 bitPos, UInt32 bitNum)
{
    PixelType val = PixelType();
    typedef typename WordTypeMaximizer<PixelType, WordType>::Type MaxType;

    const UInt32 wordBits = GetBitCount<WordType>();

    data += (bitPos / wordBits);
    bitPos = wordBits - (bitPos % wordBits);

    WordType word = 0;

    word = EndiannessConverter<WordType, Endianness>::ToHost(*data);
    MaxType firstData = MaxType(word & GetBitMask<WordType>(bitPos));
    data++;
    if (bitNum > bitPos) {
        bitNum -= bitPos;
        val += PixelType(firstData << bitNum);
    }
    else {
        val += PixelType(firstData >> (bitPos - bitNum));
        bitNum = 0;
    }

    while(bitNum >= wordBits) {
        bitNum -= wordBits;
        word = EndiannessConverter<WordType, Endianness>::ToHost(*data);
        val += PixelType(word << bitNum);
        data++;
    }

    if (bitNum > 0) {
        word = EndiannessConverter<WordType, Endianness>::ToHost(*data);
        val += PixelType(word >> (wordBits - bitNum));
        data++;
    }

    return val;
}

template<typename PixelType, typename WordType, typename Endianness>
void Access::SetBitmapBits(WordType* data, UInt32 bitPos, UInt32 bitNum, PixelType val)
{
    const UInt32 wordBits = GetBitCount<WordType>();

    data += (bitPos / wordBits);
    bitPos = wordBits - (bitPos % wordBits);

    WordType word = 0;

    if (bitNum >= bitPos) {
        bitNum -= bitPos;
        word = EndiannessConverter<WordType, Endianness>::ToHost(*data);
        word &= ~GetBitMask<WordType>(bitPos);
        word |= WordType(val >> bitNum);
        *data = EndiannessConverter<WordType, Endianness>::FromHost(word);
        data++;
    }
    else {
        word = EndiannessConverter<WordType, Endianness>::ToHost(*data);
        word &= WordType(~(GetBitMask<WordType>(bitNum) << (bitPos - bitNum)));
        word |= WordType(val << (bitPos - bitNum));
        *data = EndiannessConverter<WordType, Endianness>::FromHost(word);
        data++;
        bitNum = 0;
    }

    while(bitNum >= wordBits) {
        bitNum -= wordBits;
        word = WordType(val >> bitNum);
        *data = EndiannessConverter<WordType, Endianness>::FromHost(word);
        data++;
    }

    if (bitNum > 0) {
        word = EndiannessConverter<WordType, Endianness>::ToHost(*data);
        word &= WordType(~(GetBitMask<WordType>(bitNum) << (wordBits - bitNum)));
        word |= WordType(val << (wordBits - bitNum));
        *data = EndiannessConverter<WordType, Endianness>::FromHost(word);
        data++;
    }
}

//inverted bit accessor
template<typename PixelType, typename WordType, typename Endianness>
PixelType Access::GetBitmapBitsRightToLeft(const WordType* data, UInt32 bitPos, UInt32 bitNum)
{
    const UInt32 wordBits = GetBitCount<WordType>();

    data += bitPos / wordBits;
    bitPos = bitPos % wordBits;

    WordType word = 0;

    UInt32 firstBitCount = bitNum + bitPos;

    typedef typename WordTypeMaximizer<PixelType, WordType>::Type MaxType;
    word = EndiannessConverter<WordType, Endianness>::ToHost(*data);
    MaxType firstData = MaxType(word & GetBitMask<WordType>(firstBitCount));
    data++;
    PixelType val = PixelType(firstData >> bitPos);

    if (firstBitCount > wordBits) {
        UInt32 currentOff = wordBits - bitPos;
        while (bitNum > currentOff) {
            word = EndiannessConverter<WordType, Endianness>::ToHost(*data);
            val += PixelType(word << currentOff);
            data++;
            currentOff += wordBits;
        }
        val &= GetBitMask<PixelType>(bitNum);
    }

    return val;
}

template<typename PixelType, typename WordType, typename Endianness>
void Access::SetBitmapBitsRightToLeft(WordType* data, UInt32 bitPos, UInt32 bitNum, PixelType val)
{
    const UInt32 wordBits = GetBitCount<WordType>();

    data += bitPos / wordBits;
    bitPos = bitPos % wordBits;

    WordType word = 0;

    word = EndiannessConverter<WordType, Endianness>::ToHost(*data);
    word &= WordType(~(GetBitMask<WordType>(bitNum) << bitPos));
    word |= WordType(val << bitPos);
    *data = EndiannessConverter<WordType, Endianness>::FromHost(word);
    data++;

    if ((bitNum + bitPos) > wordBits) {
        UInt32 currentOff = wordBits - bitPos;
        while (bitNum >= (currentOff + wordBits)){
            word = WordType(val >> currentOff);
            *data = EndiannessConverter<WordType, Endianness>::FromHost(word);
            data++;
            currentOff += wordBits;
        }
        if (bitNum > currentOff) {
            bitNum -= currentOff;
            word = EndiannessConverter<WordType, Endianness>::ToHost(*data);
            word &= WordType(~GetBitMask<WordType>(bitNum));
            word |= WordType(val >> currentOff);
            *data = EndiannessConverter<WordType, Endianness>::FromHost(word);
            data++;
        }
    }
}

template<typename WordType>
inline UInt32 Access::GetBitCount()
{
    return sizeof(WordType) << 3;
}
            
template<typename WordType>
inline WordType Access::GetBitMask(UInt32 bitnum)
{
    return (bitnum < GetBitCount<WordType>()) ? ((1 << bitnum) - 1) : ~0;
}

        } //namespace Internal
    } //namespace GenericBitmapConvertor
} //namespace Candera

#endif //GENERIC_BITMAP_ACCESS_INTERNAL_H

