//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CLUT_BITMAP_EXTENSION_H
#define CLUT_BITMAP_EXTENSION_H

#include <CanderaPlatform/Device/Common/BitmapConverter/GenericBitmapFormat.h>

namespace Candera
{
    class ClutBitmapFormat : public GenericBitmapFormat
    {
    public:

        /**
          * Get the number of colors in the lookup table
          * @return number of colors in the lookup table
          */
        UInt32 GetNumberOfColors()
        {
            return m_clutData >> (5 * 4);
        }

        /**
          * Set the number of colors in the lookup table
          * @param val number of colors in the lookup table
          */
        void SetNumberOfColors(UInt32 val)
        {
            m_clutData = (m_clutData & 0x000fffff) | (val << 20);
        }

        /**
          * Get the number of bits per index (may include per-index alpha)
          * @return number of bits per index
          */
        UInt8 GetBitsPerIndex()
        {
            return UInt8(m_clutData >> (3 * 4));
        }


        /**
          * Set the number of bits per index
          * @param val number of bits per index
          */
        void SetBitsPerIndex(UInt32 val)
        {
            m_clutData = (m_clutData & 0xfff00fff) | (val << 12);
        }

        /**
          * Get the number of alpha bits in index
          * @return number of alpha bits in index
          */
        UInt8 GetAlphaBits()
        {
            return (m_clutData >> 8) % 16;
        }

        /**
          * Set the number of alpha bits in index
          * @param val number of alpha bits in index
          */
        void SetAlphaBits(UInt32 val)
        {
            m_clutData = (m_clutData & 0xfffff0ff) | (val << 8);
        }

        /**
          * Get clut generation algorithm
          * @return clut generation algorithm
          */
        UInt8 GetClutAlgorithm()
        {
            return (m_clutData >> 4) % 16;
        }

        /**
          * Set clut generation algorithm
          * @param val clut generation algorithm
          */
        void SetClutAlgorithm(UInt32 val)
        {
            m_clutData = (m_clutData & 0xffffff0f) | (val << 4);
        }

        /**
          * Get dithering
          * @return dithering
        */
        UInt8 GetDithering()
        {
            return m_clutData % 16;
        }

        /**
          * Set dithering
          * @param val dithering
        */
        void SetDithering(UInt32 val)
        {
            m_clutData = (m_clutData & 0xfffffff0) | val;
        }

        ClutBitmapFormat(UInt32 value = 0) :
            GenericBitmapFormat(),
            m_clutData(value)
        {

        }

        ClutBitmapFormat(UInt32 clutData, const GenericBitmapFormat* genericFormat):
            GenericBitmapFormat(genericFormat),
            m_clutData(clutData)
        {

        }

        private:
            UInt32 m_clutData;

    };  //class ClutBitmapFormat

}   //namespace Candera

#endif //CLUT_BITMAP_EXTENSION_H
