//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_WARP_MATRIX_H)
    #define CANDERAPLATFORM_WARP_MATRIX_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/Disposer.h>
#include <Candera/System/Mathematics/Vector2.h>

namespace Candera {

/** @addtogroup CommonDevice
 *  @{
 */

/**
 *  @brief WarpMatrix is a wrapper for a matrix used for display warping.
 *  
 *  WarpMatrix facilitates passing of the matrix to the Display object.
 *  The format of the contained data is platform specific.
 */
class WarpMatrix {
    public:
        typedef const void* ConstData;
        typedef MemoryManagement::Disposer<ConstData> ConstDataDisposer;
        typedef ConstDataDisposer::DisposerFunction ConstDataDisposerFunction;
        
        /**
         * Build an empty matrix wrapper. Data can be acquired through AcquireData.
         */
        WarpMatrix();
            
        /**
         * Build a 2D warp matrix, holding generic data.
         * The buffer is platform specific and should hold data according to platform specifications.
         * The data buffer is disposed when no longer needed. If the disposer is 0, the data is 
         * not disposed, and the application has to handle the disposal.
         * @param width             Width of the matrix.
         * @param height            Height of the matrix.
         * @param data              Pointer to associated data.
         * @param disposer          Disposer function used to dispose of the data when no longer needed.
         */
        WarpMatrix(
            Int width, 
            Int height, 
            ConstData data, 
            ConstDataDisposerFunction disposer = 0);
          
        /**
         * Destroy the matrix. Clear is called to release the data.
         */  
        ~WarpMatrix();

        /**
         * Retrieve of the data from source matrix.
         * The source matrix is stripped of its buffer, to avoid double deallocation upon 
         * destruction. i.e. The data is moved from the source to the destination.
         * If the current matrix had data store within before, the data is first cleared,
         * as if ClearData was called.
         * @param src   Matrix from which the data is taken.
         */
        void AcquireData(WarpMatrix& src);
        
        /**
         * Bring the matrix wrapper to an empty state. Any associated data is disposed,
         * if a disposer is available.
         */
        FEATSTD_LINT_CLEANUP_FUNCTION(Candera::WarpMatrix::ClearData)
        void ClearData();
        
        /**
         * Retrieve a pointer to the associated data.
         * @return      A pointer to the associated data.
         */
        ConstData GetGenericData() const { return m_data; }
         
        /**
         * Retrieve a pointer to the associated data. The data is cast to an array of
         * Vector2 objects, to be used as vertex data.
         * @return      A pointer to the associated data.
         */
        const Vector2* GetVertexData() const { 
            return FeatStd::Internal::PointerToPointer<const Vector2*>(m_data); 
        }
         
        /**
         * Retrieve a vertex from a 2D matrix.
         * This method should only be used for matrices that represent vertex data.
         * Should not be called on empty matrices and the coordinates should be
         * in space [0, width - 1]x[0, height - 1]
         * @param x     Index of the vertex in a horizontal direction.
         * @param y     Index of the vertex in a vertical direction.
         * @return      A vector representing a vertex.
         */
        Vector2 GetVertex(Int x, Int y) const { return GetVertexData()[(x + (y * m_width))]; }
        
        /**
         * Retrieve the width of a 2D matrix.
         * @return An integer representing the matrix width.
         */
        Int GetWidth() const { return m_width; }
        
        /**
         * Retrieve the height of a 2D matrix.
         * @return      An integer representing the matrix height.
         */
        Int GetHeight() const { return m_height; }

    private:
        //disallow copying of warp matrices.
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, CANDERA_LINT_REASON_NONCOPYABLE)
        FEATSTD_MAKE_CLASS_UNCOPYABLE(WarpMatrix);
        
        Int m_width; 
        Int m_height;
        ConstData m_data;
        ConstDataDisposerFunction m_disposer;
};

/** @}*/ //end of CommonDevice

}   // namespace Candera

#endif  // CANDERAPLATFORM_WARP_MATRIX_H
