//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TEXTURE_TO_IMAGE_2D_ADAPTOR_H)
#define CANDERA_TEXTURE_TO_IMAGE_2D_ADAPTOR_H

#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/Engine3D/Core/TextureImage.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera {

    /** @addtogroup Core2D
     *  @{
     */

    /**
     * @brief  Supplies an Image2D interface to a TextureImage.
     */
    class TextureImageToImage2DAdaptor : public Image2D {
        FEATSTD_TYPEDEF_BASE(Image2D);

        public:
            FEATSTD_RTTI_DECLARATION();

            typedef MemoryManagement::SharedPointer<TextureImageToImage2DAdaptor> SharedPointer;

            /**
             *  Creates an instance of this class.
             *  Use Dispose() to delete the instance and possible children, if any.
             *  @return MemoryManagement::SharedPointer to the created object.
             */
            static SharedPointer Create();

            /**
             *  Destructor
             */
            ~TextureImageToImage2DAdaptor();

            /**
             *  Assign a texture image.
             *  @param source Assigned texture image.
             *  @param pixelFormat
             */
            void SetTextureImage(const MemoryManagement::SharedPointer<TextureImage>& source, Int pixelFormat)
            {
                m_source = source;
                m_pixelFormat = pixelFormat;
            }

            /**
             *  Retrieves the surface handle provided by the texture image.
             *  @return The surface handle to the uploaded data.
             */
            virtual SurfaceHandle Get2DSurfaceHandle() const override;

            /**
             *  Retrieves the texture image associated with object.
             *  @return A shared pointer to the TextureImage object.
             */
            const MemoryManagement::SharedPointer<TextureImage>& GetTextureImage() const
            {
                return m_source;
            }

            /**
             *  Retrieves the pixel format of this object.
             *  @return The pixel format.
             */
            Int GetPixelFormat() const
            {
                return m_pixelFormat;
            }

            /**
             *  Get the attached surface's width.
             *  @return The width of the associated surface.
             */
            virtual UInt32 GetWidth() const override;

            /**
             *  Get the attached surface's height.
             *  @return The height of the associated surface.
             */
            virtual UInt32 GetHeight() const override;

        protected:
            TextureImageToImage2DAdaptor();

            virtual bool UploadInternal(LoadingHint loadingHint) override;
            virtual bool UnloadInternal(LoadingHint loadingHint) override;
            virtual void DisposeInternal() override {}

        private:

            MemoryManagement::SharedPointer<TextureImage> m_source;
            Int m_pixelFormat;
            SurfaceHandle m_surfaceHandle;
    };

    /** @} */ // end of Core2D

}

#endif //CANDERA_TEXTURE_TO_IMAGE_2D_ADAPTOR_H
