//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_SYNCHRONIZABLE_H)
    #define CANDERAPLATFORM_SYNCHRONIZABLE_H

#include <Candera/Environment.h>

namespace Candera {
/** @addtogroup CommonDevice
 *  @{
 */

/**
 *  @brief Any synchronizable object is aware about it's internal state regarding the GPU. As such, a calling method can
 *  specify the object to assure that GPUs accessing this object do this in a specific order. Further, a calling method
 *  can ask a synchronizable object to block the current thread until the object becomes synchronized.
 *
 *  A synchronizable object may become unsynchronized whenever GPU work is pending. If the work is completed by the GPU
 *  the object becomes synchronized. If a second GPU wants to work with an unsynchronized object the synchronizable
 *  interface supports specification of the dependencies.
 */
class Synchronizable {
    public:
        /// Constructor
        Synchronizable()
        {
        }

        /// Destructor
        virtual ~Synchronizable()
        {
        }

        /**
         *  Invocation of Sync will assure that the API this object is currently bound to (depends on the binding semantics)
         *  waits for this object to be in synchronized state before accessing it. This may result in having the GPU command
         *  list to wait.
         *  Further, if driver support is lacking support, it might be necessary for the implementation to block the current
         *  thread until the object is in synchronized state.
         */
        virtual void Sync() = 0;

        /**
         *  Blocks the current thread until the object is in synchronized state. Implementations should be able to determine
         *  synchronized state as good as possible because this invokation is meant to be invoked as late as possible.
         */
        virtual void WaitSync() = 0;

        /**
         *  Notify the synchronizable object that the API to which this object is bound has requested access. Such call might
         *  cause this, or other objects to become unsynchronized.
         */
        virtual void OnAccess() {}
};

/** @}*/ //end of CommonDevice
}   // namespace Candera
#endif  // CANDERAPLATFORM_SYNCHRONIZABLE_H
