//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_RENDERTARGET_H)
    #define CANDERAPLATFORM_RENDERTARGET_H

#include <CanderaPlatform/Device/Common/Base/Surface.h>
#include <CanderaPlatform/Device/Common/Base/RenderTargetListenerCollection.h>
#include <FeatStd/Util/StaticObject.h>

namespace Candera {


/** @addtogroup CommonDevice
 *  @{
 */

/**
 *  @brief RenderTarget derived objects provide a way to allow Candera APIs to access this object in a way so that the GPU
 *  can write color information to the surface. DrawTargets need to track synchronization state so that the
 *  Synchronizable interface would recognize the state this object resides in.
 *  On mult-buffered draw targets the state tracked is associated to the current background buffer.
 */
class RenderTarget : public Surface {
    public:
        /**
         * RenderTarget event listeners will get notified before a SwapBuffers occur.
         * Next is a code snippet containing a possible listener implementation that does something before a swap occur:
         *
         * @code
         * FeatStd::EventResult::Enum CustomEventListener::OnEvent(const FeatStd::Event& event)
         * {
         *     const RenderTarget::BeforeSwapEvent* beforeSwapEvent = Dynamic_Cast<const RenderTarget::BeforeSwapEvent*>(&event);
         *     if (beforeSwapEvent == 0) {
         *         return FeatStd::EventResult::Proceed; //not the reqested event
         *     }
         *     RenderTarget* renderTarget = beforeSwapEvent->GetRenderTarget(); //now we have the RenderTarget instance that triggered the event
         *     // do stuff
         * }
         * @endcode
         */
        typedef Internal::RenderTargetEventListenerCollection<RenderTarget> EventListenerCollection;
        typedef EventListenerCollection::TEvent<EventListenerCollection::BeforeSwap> BeforeSwapEvent;
        typedef EventListenerCollection::TEvent<EventListenerCollection::AfterSwap> AfterSwapEvent;

        /// Constructor
        RenderTarget() :
            m_currentRenderPass(BeforeFirstRenderPass),
            m_activeCameraCount(0),
            m_isSwapEnabled(false),
            m_isClearEnabled(false)
        {
        }

        /// Destructor
        virtual ~RenderTarget()
        {
            m_activeRenderTarget = 0;
            Internal::RenderTargetEventListenerCollection<RenderTarget>::TEvent<EventListenerCollection::Destroy> destroyEvent(this);
            GetEventSource().DispatchEvent(destroyEvent);
        }

        /**
         *  A call to BeginDraw makes the object leave the synchronized state. Within this state neither calls to another
         *  BeginDraw of the same object nor any Sync methods are allowed and would result in undefined behavior (i.e. most
         *  certainly unwanted behavior).
         */
        virtual void BeginDraw() = 0;

        /**
         *  EndDraw tells the object that the client's definition of operations have finished. Note, that at this time
         *  the state is still not necessarily synchronized as just the completeness specification is present but the
         *  GPU might still work to finish the specified task.
         *  At this point also calls to any Sync methods and BeginDraw are allowed.
         */
        virtual void EndDraw() = 0;

        /**
         *  Makes the render target active by setting the context to the associated surface.
         *  @return     true if activation was successful.
         */
        virtual bool Activate() = 0;

        /**
         *  If the draw target operates on multiple buffers the presentation buffer is exchanged with the current background
         *  buffer. Derived objects might provide additional interfaces about when buffer swapping is executed.
         */
        virtual void SwapBuffers() = 0;

        /**
         *  @brief Set buffer auto swapping enabled.
         *  Defines whether the render target swaps automatically, if all cameras have been rendered with either
         *  Renderer::RenderAllCameras or Renderer2D::RenderAllCameras. If AutoSwapEnabled is set to true, the SetSwapEnabled property
         *  of Camera and Camera2D are ignored.
         *  @param enabled specifies whether auto swapping of render target is enabled or not.
         */
        CANDERA_DEPRECATED_3_3_0("Please use a camera with swapping enabled instead.",
            void SetAutoSwapEnabled(bool enabled)
        );

        /**
         *  @brief Check if auto swapping is enabled.
         *  @return true if auto swapping is enabled, false otherwise.
         */
        CANDERA_DEPRECATED_3_3_0("Please use a camera with swapping enabled instead.",
            bool IsAutoSwapEnabled() const
        );

        /**
         * @brief Set auto clearing enabled.
         * Defines whether the render target clears automatically, before any
         *  Camera has been rendered with either Renderer::RenderAllCameras or
         *  Renderer2D::RenderAllCameras. As oposed to the AutoSwap flag, the
         *  AutoClear does not invalidate Camera clearing setting.
         *  @param enabled specifies whether auto clearing of render target is enabled or not.
         */
        CANDERA_DEPRECATED_3_3_0("Please use a camera with clearing enabled instead.",
            void SetAutoClearEnabled(bool enabled)
        );

        /**
         *  @brief Check if auto clearing is enabled.
         *  @return true if auto clearing is enabled, false otherwise.
         */
        CANDERA_DEPRECATED_3_3_0("Please use a camera with clearing enabled instead.",
            bool IsAutoClearEnabled() const
        );

        /** Calls a BeginDraw and caches the active draw target, which can be further retrieved using GetActiveRenderTarget method.
        * @param renderTarget instance of the current draw target.
        */
        void SafeBeginDraw(RenderTarget* renderTarget) { BeginDraw(); m_activeRenderTarget = renderTarget; }

        /**
        * @brief Calls an EndDraw and sets the cached draw target to 0.
        */
        void SafeEndDraw() { m_activeRenderTarget = 0; EndDraw(); };

        /** Returns a valid pointer to the current active draw target if called between a SafeBeginDraw() / SafeEndDraw() sequence, NULL otherwise.
        * @return a pointer to the cached draw target. <em>NULL</em> if not between a SafeBeginDraw() / SafeEndDraw() sequence.
        */
        static RenderTarget* GetActiveRenderTarget() { return m_activeRenderTarget; }

        /**
         * Add a listener for RenderTarget generated events.
         *
         * Each listener is added only once, and it will be removed after the first call to RemoveRenderTargetEventListener.
         *
         * @param listener Listener for RenderTarget events.
         * @see Candera::RenderTarget::BeforeSwapEvent
         */
        void AddEventListener(FeatStd::EventListener* listener) { GetEventSource().AddEventListener(listener);}

        /**
        * Remove existing listener for RenderTarget generated events.
        *
        * Removing a listener will delete if from the listener list, regardless of the number of times it was previously added.
        *
        * @param listener Listener for RenderTarget events.
        */
        void RemoveEventListener(FeatStd::EventListener* listener) { GetEventSource().RemoveEventListener(listener);}

protected:
    enum CurrentRenderPass {
        BeforeFirstRenderPass,  ///< Before first render pass.
        DuringFirstRenderPass,  ///< During first render pass.
        AfterFirstRenderPass,   ///< After first render pass.
        AfterLastRenderPass     ///< After last render pass.
    };
    CurrentRenderPass m_currentRenderPass;
    UInt32 m_activeCameraCount;

    void NotifyListenersOnBeforeSwapBuffersCalled() { 
       Internal::RenderTargetEventListenerCollection<RenderTarget>::TEvent<EventListenerCollection::BeforeSwap> beforeSwapEvent(this);
        GetEventSource().DispatchEvent(beforeSwapEvent);
    }
    void NotifyListenersOnAfterSwapBuffersExecuted() { 
       Internal::RenderTargetEventListenerCollection<RenderTarget>::TEvent<EventListenerCollection::AfterSwap> afterSwapEvent(this);
        GetEventSource().DispatchEvent(afterSwapEvent);
    }

private:
    template <typename T> friend struct Internal::InternalEventSourceAccess;
    static FeatStd::EventSource& GetEventSource()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::EventSource, s_eventSource);
        return s_eventSource;
    }

    bool m_isSwapEnabled;
    bool m_isClearEnabled;
    static RenderTarget* m_activeRenderTarget;

};

/** @}*/ //end of CommonDevice
}   // namespace Candera

#endif  // CANDERAPLATFORM_RENDERTARGET_H
