//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_DISPLAY_H)
    #define CANDERAPLATFORM_DISPLAY_H

#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>
#include <CanderaPlatform/Device/Common/Base/DeviceMetaInfo.h>
#include <CanderaPlatform/Device/Common/Base/DisplayOrientation.h>
#include <CanderaPlatform/Device/Common/Base/WarpMatrix.h>

#include <CanderaPlatform/Device/Common/Internal/WarpingProperties.h>

namespace Candera {

class SimulatedDisplay;
class DevicePackageInterface;
class Rectangle;
namespace Internal {
    class DisplayAccess;
}

/** @addtogroup CommonDevice
 *  @{
 */

/**
 *  @brief Display is a wrapper class for the native display. It serves as host for render targets
 *  and allows association with Context Resource Pools.
 */

class Display : public DeviceMetaInfoHost {
    public:
        /**
         * @brief Structure used to set the display configuration.
         *
         * If only width, height, refreshRate and colorBits are provided,
         * an attempt is made to compute the rest of the parameters.
         * The rest of the parameters shall be 0.
         *
         * Pixel clock can be computed if set to 0 if the other parameters
         * are supplied.
         *
         * The values width, height, refreshRate and colorBits that are
         * supported for the default computation is platform specific, and
         * the computation may fail resulting in failure to initialize the
         * display.
         *
         * Some platforms will ignore some of the parameters.
         */
        struct CommonSettings {
            Int width;                  /*!< Horizontal Display Period - Illuminated area */
            Int hps;                    /*!< Number of the dot when the sync pulse starts */
            Int hpe;                    /*!< Number of the dot when the sync pulse ends */
            Int hpt;                    /*!< Total horizontal */

            Int height;                 /*!< Vertical display period - Illuminated area */
            Int vps;                    /*!< Vertical sync pulse start position */
            Int vpe;                    /*!< Vertical sync end position */
            Int vpt;                    /*!< Total vertical lines */

            Int refreshRate;            /*!< Vertical refresh rate */

            Int colorBits;              /*!< Total color bits */

            Float pixelClock;           /*!< Pixel clock frequency. Set to 0 to compute it automatically. */
            bool invertDisplayClock;    /*!< Display clock output signal is inverted when this is true. */

            Int hbp;                    /*!< Horizontal back porch. */
            Int vbp;                    /*!< Vertical back porch. */
        };

        /**
         * Retrieves the associated Context Resource Pool.
         * The associated Context Resource Pool is the one used for Render
         * Targets pertaining to this display.
         * If no context resource pool is available for this display, the
         * display render targets created for this display each get their own
         * context resource pool.
         * @return  Pointer to the associated Context Resource Pool.
         */
        ContextResourcePool* GetDefaultContextResourcePool() const;

        /**
         * Retrieve the hardware display identifier.
         * @return  The hardware dependent display identifier as used with
         *          DevicePackageInterface::CreateDisplay.
         */
        Int GetDisplayId() const { return m_displayId; }

        /**
         * Create the video resources required for manipulating a display.
         * @param settings  Common parameters used in initialization of a display.
         * @return          true if everything uploaded successfully, false otherwise.
         */
        bool Upload(const CommonSettings& settings);

        /**
         * An alternative to Upload. Native resources are created outside, and only
         * a handle is attached to the display object. This function creates the associations
         * needed internally by the graphic package. (e.g. a simulated display).
         * @param intendedWidth     Intended width display. (in most cases it should correspond to actual width of the display)
         * @param intendedHeight    Intended height display. (in most cases it should correspond to actual height of the display)
         * @param handle            Native handle of the display.
         * @return                  true if everything uploaded successfully, false otherwise.
         */
        virtual bool AttachNativeHandle(Int intendedWidth, Int intendedHeight, void* handle) = 0;

        /**
         * Unload all video resources and/or the associations created by AttachHandle.
         */
        void Unload();

        /**
         * Retrieve a pointer to a simulated display object.
         * @return  A pointer simulated display object, or 0 if a simulation is not available.
         */
        virtual SimulatedDisplay* ToSimulatedDisplay() = 0;

        /**
         * Retrieve a pointer to the native handle attached by AttachNativeHandle, or created
         * by Upload.
         * @return  A native display handle, or 0 if none is available.
         */
        virtual void* GetNativeHandle() = 0;

        /**
         * Determines whether this display configuration is valid.
         * @return  true if the display is configured correctly.
         */
        virtual bool IsValid() { return true; }

        /**
         * Set the matrix used for display warping.
         * The warping matrix is used when the display is uploaded.
         * SetWarpMatrix calls WarpMatrix::AcquireData internally,
         * so the input warpMatrix is empty when the method returns.
         * Upload or, if already uploaded, ApplyChanges must be called
         * to have the display used this warp matrix.
         * Setting an empty warp matrix will cause the default
         * matrix to be used, which will come with all the performance penalties
         * of warping, but will display as not warped.
         * To disable warping call SetWarpingEnabled(false).
         * Warping is disabled by default.
         * @param warpMatrix    Matrix that is set to the display.
         * @param index    Index for the corresponding warp matrix, if more than one is provided.
         */
        void SetWarpMatrix(WarpMatrix& warpMatrix, UInt index = 0) { m_warpingProperties.SetWarpMatrix(warpMatrix, index); }

        /**
         * Get the matrix used for display warping.
         * @param index     Index of the requested warp matrix, when more than one is available.
         * @return  The matrix that was set to the display.
         */
        const WarpMatrix& GetWarpMatrix(UInt index = 0) const { return m_warpingProperties.GetWarpMatrix(index); }
        WarpMatrix& GetWarpMatrix(UInt index = 0) { return m_warpingProperties.GetWarpMatrix(index); }

        /**
        * Reset the matrix at index
        * @param index    Index for the warp matrix that will be reset
        */
        void ResetWarpMatrix(UInt index) { m_warpingProperties.ResetWarpMatrix(index); }

        /**
        *  Sets a warp matrix weight.
        *  @param index of the corresponding warp matrix.
        *  @param weight of the warp matrix.
        */
        void SetWarpMatrixWeight(UInt index, Float weight) { m_warpingProperties.SetWarpMatrixWeight(index, weight); }

        /**
        *  Retrieves the weight of warp matrix.
        *  @param index     Index of the corresponding warp matrix, when more than one is available.
        *  @return      The weight of the requested warp matrix.
        */
        Float GetWarpMatrixWeight(UInt index) const { return m_warpingProperties.GetWarpMatrixWeight(index); }

         /**
         * Sets the bounding rectangle used to determine the area of the warping texture to use.
         * @param warpImageBounds The bounds of the warping texture to map to mesh.
         */
        void SetWarpImageBounds(Rectangle& warpImageBounds) { m_warpingProperties.SetWarpImageBounds(warpImageBounds); }
        
        /**
         * Retrieves the bounds of the warping image to map.
         * @return The bounds of the warping image to map.
         */
        const Rectangle& GetWarpImageBounds() const { return m_warpingProperties.GetWarpImageBounds(); }

        /** 
         * Sets the orientation of the display to use during warping.
         * @param orientation The orientation to use during warping.
         */        
        void SetWarpOrientation(DisplayOrientation::Enum orientation) { m_warpingProperties.SetDisplayOrientation(orientation); }
        
        /** 
         * Retrieves the orientation of the display to use during warping.
         * @return The orientation of the display currently used during warping.
         */
        DisplayOrientation::Enum GetWarpOrientation() const { return m_warpingProperties.GetDisplayOrientation(); }

        /**
         * Enables or disables warping. The display needs to be unloaded
         * before a change to this flag can take effect.
         * @param isEnabled True to enable, and false to disable warping. Default value is false.
         * @return True if warping can be enabled at this time. False otherwise.
         */
        bool SetWarpingEnabled(bool isEnabled);

        /**
         * Returns the value set by SetWarpingEnabled, or the default 'false'.
         * @return Whether warping is enabled or not.
         */
        bool IsWarpingEnabled() const { return m_warpingProperties.IsWarpingEnabled(); }

    protected:
        /**
         * Constructor. Upon construction the active context resource pool is
         * associated with the display. Creation is only available via
         * DevicePackageInterface::CreateDisplay.
         * @param displayId     A hardware dependent display identifier starting with 0.
         *                      The meaning of the identifiers are specific to the implementation.
         *                      This should match the id used by the DevicePackageInterface::CreateDisplay,
         *                      since it might be used by Upload.
         */
        Display(Int displayId);

        /**
         * Destructor. Virtual to facilitate platform specific implementations.
         * Destruction is only available via DevicePackageInterface::DestroyDisplay.
         */
        virtual ~Display() { m_pool = 0; }

        /**
         * Set the value of the warp matrix dirty flag.
         * This flag is set to true by SetWarpMatrix, and should be reset by
         * Upload and ApplyChanges.
         * @param flag True to use the matrix, false to ignore.
         */
        void SetWarpMatrixDirtyFlag(bool flag) { m_warpingProperties.SetWarpMatrixDirtyFlag(flag); }

        /**
         * Get the value of the warp matrix dirty flag.
         * @return True to use the matrix, false to ignore.
         */
        bool IsWarpMatrixDirty() const { return m_warpingProperties.IsWarpMatrixDirty(); }

        /**
         * Get the warping properties of this display.
         * @return Warping properties.
         */
        WarpingProperties& GetWarpingProperties() { return m_warpingProperties; }
        const WarpingProperties& GetWarpingProperties() const { return m_warpingProperties; }

        /**
         * Get if this display was successfully uploaded.
         * @return Returns true if the display was successfully uploaded.
         */
        bool IsUploaded() const { return m_isUploaded; };

        /**
         * Needs to be implemented by derived classes to define the behavior of Upload.
         * For mode details, see Upload.
         * @param settings  Common parameters used in initialization of a display.
         * @return          true if everything uploaded successfully, false otherwise.
         */
        virtual bool UploadInternal(const CommonSettings& settings) = 0;

        /**
         * Needs to be implemented by derived classes to define the behavior of Unload.
         * For mode details, see Unload.
         */
        virtual void UnloadInternal() = 0;

    private:
        friend class DevicePackageInterface;
        friend class Internal::DisplayAccess;

        //disallow copying of displays.
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, CANDERA_LINT_REASON_NONCOPYABLE)
        FEATSTD_MAKE_CLASS_UNCOPYABLE(Display);
        
        bool m_isUploaded;

        // Used as workaround in simulation to override the use of multiple
        // context resource pools.
        ContextResourcePool* m_pool;

#ifdef CANDERA_DEPRECATED_3_0_0
        ContextResourcePool* m_fallbackPool;
#else
#error Remove this!
#endif

        Int m_displayId;

        WarpingProperties m_warpingProperties;
};

/** @}*/ //end of CommonDevice

namespace Internal {
/**
 * @brief Internal workaround for deprecated interfaces in Display.
 */
class DisplayAccess 
{
public:
    /**
     * Retrieve the context resource pool set by the user.
     * @param display Source display.
     * @return Pointer to a Context Resource Pool, 0 is no pool is set.
     */
    static ContextResourcePool* GetUserContextResourcePool(const Display* display);
    /**
     * Set user context resource pool.
     * @param display Destination display.
     * @param pool Pointer to a Context Resource Pool, 0 to clear current value.
     * @return True if setting was successful.
     */
    static bool SetUserContextResourcePool(Display* display, ContextResourcePool* pool);
    /**
     * Retrieve Context resource pool used as fallback when user resource pool
     * is 0.
     * @param display Source display.
     * @return Pointer to a Context Resource Pool, guaranteed to be non-zero.
     */
    static ContextResourcePool* GetFallbackContextResourcePool(const Display* display);
    /**
     * Retrieve the effective ContextResourcePool of the display. This is
     * the context resource pool that should be used by objects that inherit
     * from the display.
     * @param display Source display.
     * @return Pointer to a Context Resource Pool, guaranteed to be non-zero.
     */
    static ContextResourcePool* GetEffectiveContextResourcePool(const Display* display);
};
}

}   // namespace Candera

#endif  // CANDERAPLATFORM_DISPLAY_H
