//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DIRECTTEXTURE_IMAGE_H)
#define CANDERA_DIRECTTEXTURE_IMAGE_H

#include <FeatStd/Diagnostics/Measurable.h>

#include <Candera/Engine3D/Core/TextureImage.h>
#include <CanderaPlatform/Device/Common/Base/ImageSource3D.h>

#if defined(CANDERA_DEVICE_APOLLOLAKE) && (!defined(CANDERA_OS_QNX))
    #include <CanderaPlatform/Device/ApolloLake/DRM/ApolloLakeDrmBuffer.h>
#endif // defined(CANDERA_DEVICE_APOLLOLAKE)

namespace Candera {
/** @addtogroup CommonDevice
 *  @{
 */

    class Texture;

/**
* @brief  Direct texture images are texture images that allow direct access by
* the CPU to their buffer.
*
* This class is useful when the content of the texture needs to be updated
* frequently by CPU, for instance when mapping live video.
 */
class DirectTextureImage : public TextureImage, public FeatStd::Diagnostics::Measurable
{
    friend class RenderDevice;
    FEATSTD_TYPEDEF_BASE(TextureImage);

    public:
        /**
         * @brief Maximal number of supported image planes.
         */
        enum { c_maxPlanesCount = 4 };

        /**
         * @brief Enumeration usable as format with direct textures.
         *
         * Each format describes the way the pixels are formated, packed and
         * distributed among separate buffers, that are returned when Locking
         * the texture image.
         */
        enum Format {
            FormatYV12,   ///< Planar YV12 4:2:0 texture; buffers: Yplane, Vplane, Uplane.
            FormatI420,
            FormatNV12,   ///< Planar NV12 4:2:0 texture; buffers: Yplane, UVplane.
            FormatNV21,   ///< Planar NV21 4:2:0 texture; buffers: Yplane, VUplane.
            FormatYUY2,   ///< Planar packed 4:2:2 texture; buffers: packed YUV texture.
            FormatUYVY,   ///< Planar packed 4:2:2 texture; buffers: packed YUV texture.
            FormatRGBA,   ///< Unpacked RGBA texture format, 32 bit.
            FormatBGRA,   ///< Unpacked BGRA texture format, 32 bit.
            FormatAlpha,  ///< Alpha format, 8 bit.
            FormatRGB565, ///< Unpacked RGB565 texture format, 16 bit.
            FormatRGB
        };

        /**
         *  Creates an instance of this class.
         *  @return   A MemoryManagement::SharedPointer which manages the
         *            lifetime of the instance.
         */
        static MemoryManagement::SharedPointer<DirectTextureImage> Create();

        /**
         *  Destructor
         */
        virtual ~DirectTextureImage();

        /**
         *  Lock this object.
         *  Locking an object will cause it to fail to activate until it is
         *  Unlocked. Locking provides memory pointers which may be used to
         *  change the content of the texture image. The format of the buffers
         *  is the one specified when setting up the texture. See Format.
         *  If the Texture was not successfully uploaded, locking will fail.
         *  Lock should be called as late after drawing as possible.
         *  @param buffers [out] Buffer that will be populated with pointers.
         *                       This array should be able to hold at least
         *                       4 pointers. The non-null pointers shall be used
         *                       to directly update the content of texture.
         *  @return True if successful, false otherwise.
         */
        bool Lock(void* buffers[]);

        /**
         *  Unlock this object.
         *  Unlocking will cause the pointers returned by Lock to become invalid and will let the GPU know that the texture data has changed.
         *  Using the pointer after Unlock was called will result in undefined
         *  behavior.
         *  Once unlocked the texture image may be used for rendering.
         *  Unlock should be called as early before rendering as possible.
         */
        void Unlock();

        /**
         * Determine whether this DirectTextureImage object is locked or not.
         * @return True if locked, false if not locked.
         */
        bool IsLocked() const { return m_isLocked; }

        /**
         *  Invalidate the texture data of this DirectTextureImage.
         * Invalidation will let the GPU know that the texture data has changed.
         * Applications that provide a logical or physical address for the texture data should call this function when the data has changed.
         * If the buffer was allocated by the direct texture image itself Lock/Unlock should be used instead.
         */
        void Invalidate();

        /**
         *  Sets the width of the texture image. Width must be 16 pixel aligned.
         *  The default value of this parameter is 64.
         *  Call this function before Upload. If called on uploaded image it
         *  will fail.
         *  @param width Width of the texture image.
         *  @return True if successful, false if Uploaded.
         */
        bool SetWidth(Int width);

        /**
         *  Sets the height of the texture image.
         *  The default value of this parameter is 64.
         *  @param height Height of the texture image.
         *  @return True if successful, false if Uploaded.
         */
        bool SetHeight(Int height);

        /**
         *  Sets the format of the texture image.
         *  The default value of this parameter is FormatYUY2.
         *  @param format Format of the texture image.
         *  @return True if successful, false if Uploaded.
         */
        bool SetFormat(Format format);

        /**
         *  Retrieves width set to this Texture object.
         *  @return Width of the texture image.
         */
        Int GetWidth() const { return m_width; }

        /**
         *  Retrieves height set to this Texture object.
         *  @return Height of the texture image.
         */
        Int GetHeight() const { return m_height; }

        /**
         *  Retrieves format set to this Texture object.
         *  @return Format of the texture image.
         */
        Format GetFormat() const { return m_format; }

        /**
         * Retrieves offset for each used plane according to format of this texture image.
         * @param offset Output array of planes offsets, with values relative to first plane. First offset is always zero, and was left out. Only elements for first used planes are filled.
         * @param totalSize Output parameter filled with sum of planes sizes.
         * @return Number of planes associated with the format, or 0 on error.
         */
        UInt GetPlanesOffsetsAndTotalSize(UInt offset[c_maxPlanesCount - 1], UInt& totalSize) const;

        /**
         * Retrieves pitch in pixels for each used plane according to format of this texture image.
         * @param pitch Output parameter, array of planes pitches in pixels. Only elements for first used planes are filled.
         * @return Number of planes associated with the format, or 0 on error.
         */
        UInt GetPlanesPitches(UInt pitch[c_maxPlanesCount]) const;

        /**
         *  Defines if MipMap-chain shall be attached to TextureImage.
         *  When enabled the MipMap-chain is generated when Unlock is called.
         *  By default this is disabled.
         *  @param enableMipMapping True to enable, false to disable MipMapping.
         */
        bool SetMipMappingEnabled(bool enableMipMapping);

        /**
         *  Sets an array of logical addresses of the application-defined texture buffers.
         *  The logical address array is set if a buffer is provided and the driver does not have to allocate memory for the direct texture.
         *  Usually only the first element of the array is needed, but a set of pointers is required for some YUV formats.
         *  Alternatively, the first element of the array might contain a target specific buffer handle. The render device will use this handle to obtain the logical addresses. The buffer is assumed to be correctly initialized.
         *
         *  When using buffers not allocated by DirectTexture itself, Invalidate() needs to be called when the data in the texture buffer is changed.
         *
         *  The default value is {0, 0, 0, 0}.
         *  Logical addresses must be 64 bit (8 byte) aligned.
         *  Unused addresses should be set to 0.
         *  @param logicalAddress the pointer array, needs to hold at least four pointers.
         */
        void SetLogicalAddress(void* const logicalAddress[4]);

        /**
         *  This is used to get the logical memory addresses of the application-defined texture buffers.
         *  @param logicalAddress the pointer array, needs to hold at least four pointers.
         */
        void GetLogicalAddress(void* logicalAddress[4]) const;

        /**
         *  Sets an array of physical addresses of the application-defined buffers to
         *  the texture. Should be set to the default value of {~0U, ~0U, ~0U, ~0U} if no physical address has been provided.
         *
         *  When using buffers not allocated by DirectTexture itself, Invalidate() needs to be called when the data in the texture buffer is changed.
         *
         *  Usually only the first element of the array is needed, but a set of pointers is required for some YUV formats.
         *  @param physicalAddress the physical address array, needs to hold at least four pointers.
         */
        void SetPhysicalAddress(UInt const physicalAddress[4]);

        /**
         *  This is used to get the physical addresses of the application-defined buffers to
         *  the texture image.
         *  @param physicalAddress the physical address array, needs to hold at least four pointers.
         */
        void GetPhysicalAddress(UInt physicalAddress[4]) const;

        // Overridden function of TextureImage.
        virtual Handle GetVideoMemoryHandle() const;
        // Overridden function of TextureImage.
        virtual bool Activate(UInt unit);
        using Base::Activate;
        // Overridden function of TextureImage.
        virtual bool IsMipMappingEnabled() const { return m_mipMappingEnabled; }
        // Overridden function of TextureImage.
        virtual ImageSource3D* ToImageSource3D() { return &m_imageSource3DInstance; }
        // Overridden function of TextureImage.
        virtual TextureTargetType GetTextureTargetType() const { return TextureImage::Texture2D; }
        // Overridden function of Measurable.
        virtual UInt GetSize() const;

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Overridden function of DeviceObject.
        virtual bool UploadInternal(LoadingHint loadingHint);

        // Overridden function of DeviceObject.
        virtual bool UnloadInternal(LoadingHint loadingHint);

        // Overridden function of DeviceObject.
        virtual void DisposeInternal();

        // Protected because only ::Create() can create an instance of this class. Derived classes are able to access constructor, though.
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::DirectTextureImage::DirectTextureImage, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        DirectTextureImage();
        DirectTextureImage(const DirectTextureImage& material);

    private:
        DirectTextureImage& operator = (const DirectTextureImage& material);

        /**
         *  This is used by the render device to set the texture handle to
         *  this texture image.
         *  @param handle The texture handle
         */
        void SetVideoMemoryHandle(Handle handle);

        /**
         *  This is used by the render device to set the texture handle to
         *  this texture image.
         *  @param address the texture handle
         */
        void SetVideoMemoryAddress(void* const address[4]);

        /**
         *  This is used by the render device to get the texture handle to
         *  this texture image.
         *  @param address the texture handle
         */
        void GetVideoMemoryAddress(void* address[4]) const;

#if defined(CANDERA_DEVICE_APOLLOLAKE) && (!defined(CANDERA_OS_QNX))
        /**
         *  This is used by the render device to get DRM buffer associated with this texture image.
         *  @return SharedPointer to associated DRM buffer.
         */
        Internal::ApolloLakeDrmBuffer::SharedPointer GetDrmBuffer() const { return m_drmBuffer; }

        /**
         *  This is used by the render device to set the DRM buffer to this texture image.
         *  @param buffer SharedPointer to DRM buffer to set.
         */
        void SetDrmBuffer(Internal::ApolloLakeDrmBuffer::SharedPointer buffer) { m_drmBuffer = buffer; }

        /**
         *  This is used by the render device to get EGL image handle associated with this texture image.
         *  @return Associated EGL image handle.
         */
        void* GetEglImage() const { return m_eglImage; }

        /**
         *  This is used by the render device to set the EGL image handle to this texture image.
         *  @param eglImage EGL image handle to set.
         */
        void SetEglImage(void* eglImage) { m_eglImage = eglImage; }
#endif // defined(CANDERA_DEVICE_APOLLOLAKE) && !defined(CANDERA_OS_QNX)

        /**
         *  Wrapper class which is exposed by ToImageSource3D function.
         */
        class ImageSource3DInstance : public ImageSource3D
        {
            friend class DirectTextureImage;

            public:
                ImageSource3DInstance() : m_directTextureImage(0) { }
                ~ImageSource3DInstance() { m_directTextureImage = 0; }

                virtual Handle GetVideoMemoryHandle() const { return (m_directTextureImage != 0) ? m_directTextureImage->GetVideoMemoryHandle() : 0; }
                virtual bool IsMipMappingEnabled() const { return (m_directTextureImage != 0) ? m_directTextureImage->IsMipMappingEnabled() : false; }
                virtual Int GetHeight() const { return (m_directTextureImage != 0) ? static_cast<Int>(m_directTextureImage->GetHeight()) : 0; }
                virtual Int GetWidth() const { return (m_directTextureImage != 0) ? static_cast<Int>(m_directTextureImage->GetWidth()) : 0; }
                virtual void Sync() {}
                virtual void WaitSync() {}

            private:
                const DirectTextureImage* m_directTextureImage;

                void SetDirectTextureImage(const DirectTextureImage* image) { m_directTextureImage = image; }
        };

        bool m_isLocked;
        bool m_mipMappingEnabled;

        Int m_width;
        Int m_height;
        Format m_format;

        Handle m_videoMemoryHandle[CANDERA_MAX_CONTEXT_COUNT];
        void* m_videoMemoryAddress[CANDERA_MAX_CONTEXT_COUNT][4];

        void* m_logicalAddress[4];
        UInt m_physicalAddress[4];

        ImageSource3DInstance m_imageSource3DInstance;

#if defined(CANDERA_DEVICE_APOLLOLAKE) && (!defined(CANDERA_OS_QNX))
        Internal::ApolloLakeDrmBuffer::SharedPointer m_drmBuffer;
        void* m_eglImage;
#endif // defined(CANDERA_DEVICE_APOLLOLAKE)
};

/** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_TEXTURE_IMAGE_H
