//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "DirectTextureImage.h"

#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/Engine3D/Core/Renderer.h>

namespace Candera {
    using namespace Diagnostics;

FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

DirectTextureImage::DirectTextureImage() :
    m_isLocked(false),
    m_mipMappingEnabled(false),
    m_width(64),
    m_height(64),
    m_format(FormatYUY2)
#if defined(CANDERA_DEVICE_APOLLOLAKE) && (!defined(CANDERA_OS_QNX))
,   m_eglImage(0)
#endif // defined(CANDERA_DEVICE_APOLLOLAKE)
{
    MemoryPlatform::Set(m_videoMemoryHandle, 0, sizeof(m_videoMemoryHandle));
    MemoryPlatform::Set(m_videoMemoryAddress, 0, sizeof(m_videoMemoryAddress));
    MemoryPlatform::Set(m_logicalAddress, 0, sizeof(m_logicalAddress));
    MemoryPlatform::Set(m_physicalAddress, 0xFF, sizeof(m_physicalAddress));
    m_imageSource3DInstance.SetDirectTextureImage(this);
}

MemoryManagement::SharedPointer<DirectTextureImage> DirectTextureImage::Create()
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
    DirectTextureImage* ptr = FEATSTD_NEW(DirectTextureImage)();
    if (ptr == 0) {
        FEATSTD_LOG_ERROR("Texture image create failed, out of memory.");
    }
    MemoryManagement::SharedPointer<DirectTextureImage> sharedPointer(ptr);
    return sharedPointer;
}

CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1579, "direct data is disposed by the disposer; the disposer has external lifetime")
DirectTextureImage::~DirectTextureImage()
{
    static_cast<void>(Unload(ForceAll));
}

bool DirectTextureImage::Lock(void* buffers[])
{
    FEATSTD_DEBUG_ASSERT(buffers != (void*)0);

    if ((!IsUploaded()) || m_isLocked) {
        return false;
    }
    bool success = RenderDevice::LockDirectTextureImage(*this);
    if (success) {
        const Int poolIndex = ContextResourcePool::GetActive().GetIndex();
        buffers[0] = m_videoMemoryAddress[poolIndex][0];
        buffers[1] = m_videoMemoryAddress[poolIndex][1];
        buffers[2] = m_videoMemoryAddress[poolIndex][2];
        buffers[3] = m_videoMemoryAddress[poolIndex][3];

        m_isLocked = true;
    }
    else {
        buffers[0] = 0;
        buffers[1] = 0;
        buffers[2] = 0;
        buffers[3] = 0;
    }
    return success;
}

void DirectTextureImage::Unlock()
{
    if ((!IsUploaded()) || (!m_isLocked)) {
        return;
    }
    if (RenderDevice::UnlockDirectTextureImage(*this, 0)) {
        m_isLocked = false;
    }
}

void DirectTextureImage::Invalidate() {
    if (!IsUploaded()) {
        return;
    }
    static_cast<void>(RenderDevice::UnlockDirectTextureImage(*this, 0));
}

bool DirectTextureImage::SetWidth(Int width)
{
    if (IsUploaded()) {
        return false; // disallow changing width when uploaded
    }
    m_width = width;
    return true;
}

bool DirectTextureImage::SetHeight(Int height)
{
    if (IsUploaded()) {
        return false; // disallow changing height when uploaded
    }
    m_height = height;
    return true;
}

bool DirectTextureImage::SetFormat(Format format)
{
    if (IsUploaded()) {
        return false; // disallow changing format when uploaded
    }
    m_format = format;
    return true;
}

void DirectTextureImage::GetLogicalAddress(void* logicalAddress[4]) const
{
    logicalAddress[0] = m_logicalAddress[0];
    logicalAddress[1] = m_logicalAddress[1];
    logicalAddress[2] = m_logicalAddress[2];
    logicalAddress[3] = m_logicalAddress[3];
}

void DirectTextureImage::SetLogicalAddress(void* const logicalAddress[4])
{
    m_logicalAddress[0] = logicalAddress[0];
    m_logicalAddress[1] = logicalAddress[1];
    m_logicalAddress[2] = logicalAddress[2];
    m_logicalAddress[3] = logicalAddress[3];
}

void DirectTextureImage::GetPhysicalAddress(UInt physicalAddress[4]) const
{
    physicalAddress[0] = m_physicalAddress[0];
    physicalAddress[1] = m_physicalAddress[1];
    physicalAddress[2] = m_physicalAddress[2];
    physicalAddress[3] = m_physicalAddress[3];
}

void DirectTextureImage::SetPhysicalAddress(UInt const physicalAddress[4])
{
    m_physicalAddress[0] = physicalAddress[0];
    m_physicalAddress[1] = physicalAddress[1];
    m_physicalAddress[2] = physicalAddress[2];
    m_physicalAddress[3] = physicalAddress[3];
}

Handle DirectTextureImage::GetVideoMemoryHandle() const
{
    return m_videoMemoryHandle[ContextResourcePool::GetActive().GetIndex()];
}

bool DirectTextureImage::Activate(UInt unit)
{
    if (m_isLocked) {
        return false;
    }
    return Base::Activate(unit);
}

UInt DirectTextureImage::GetPlanesOffsetsAndTotalSize(UInt offset[c_maxPlanesCount - 1], UInt& totalSize) const
{
    UInt planesCount = 0;

    FEATSTD_DEBUG_ASSERT(m_width >= 0 && m_height >= 0);
    const UInt size = static_cast<UInt>(m_width) * static_cast<UInt>(m_height);

    switch (m_format) {
    case FormatNV12:
    case FormatNV21: {
        planesCount = 2;
        offset[0] = size;
        const UInt width2 = (static_cast<UInt>(m_width) + 1) / 2;
        const UInt height2 = (static_cast<UInt>(m_height) + 1) / 2;
        const UInt size2 = width2 * height2;
        totalSize = size + 2 * size2;
        break; }
    case FormatYUY2:
    case FormatUYVY:
        planesCount = 1;
        totalSize = size * 2;
        break;
    case FormatRGBA:
    case FormatBGRA:
        planesCount = 1;
        totalSize = size * 4;
        break;
    case FormatAlpha:
    case FormatYV12:
    case FormatI420:
    default:
        FEATSTD_LOG_ERROR("Unknown or unsupported DirectTexureImage format %u.", m_format);
        break;
    }

    return planesCount;
}

UInt DirectTextureImage::GetPlanesPitches(UInt pitch[c_maxPlanesCount]) const
{
    UInt planesCount = 0;

    FEATSTD_DEBUG_ASSERT(m_width >= 0);
    switch (m_format) {
    case FormatNV12:
    case FormatNV21:
        planesCount = 2;
        pitch[0] = static_cast<UInt>(m_width);
        pitch[1] = pitch[0];
        break;
    case FormatYUY2:
    case FormatUYVY:
        planesCount = 1;
        pitch[0] = static_cast<UInt>(m_width) * 2;
        break;
    case FormatRGBA:
    case FormatBGRA:
        planesCount = 1;
        pitch[0] = static_cast<UInt>(m_width);
        break;
    case FormatAlpha:
    case FormatYV12:
    case FormatI420:
    default:
        FEATSTD_LOG_ERROR("Unknown or unsupported DirectTexureImage format %u.", m_format);
        break;
    }

    return planesCount;
}


bool DirectTextureImage::SetMipMappingEnabled(bool enableMipMapping)
{
    if (IsUploaded()) {
        return false; // disallow changing format when uploaded
    }
    m_mipMappingEnabled = enableMipMapping;
    return true;
}

UInt DirectTextureImage::GetSize() const
{
    Int nom = 0;
    const Int denom = 2;
    switch (m_format) {
        case FormatAlpha: // 8 bit
            nom = 2;
            break;
        case FormatYV12:
        case FormatNV12:
        case FormatNV21: //4:2:0
        case FormatI420:
        case FormatRGB:
            nom = 3;
            break;
        case FormatYUY2:
        case FormatUYVY: //4:2:2
            nom = 4;
            break;
        case FormatRGBA:
        case FormatBGRA: // 32 bit
            nom = 8;
            break;
        case FormatRGB565: // 16 bit
            nom = 4;
            break;
        default:
            break;
    }
    return static_cast<UInt>((m_width * m_height * nom) / denom);
}

bool DirectTextureImage::UploadInternal(LoadingHint loadingHint)
{
    return Renderer::UploadDirectTextureImage(*this, 0, loadingHint);
}

bool DirectTextureImage::UnloadInternal(LoadingHint loadingHint)
{
    m_isLocked = false;
    return Renderer::UnloadDirectTextureImage(*this, loadingHint);
}

void DirectTextureImage::DisposeInternal()
{
    // Reset the adresses, including logical and physical
    void* address[4] = { 0, 0, 0, 0 };
    SetVideoMemoryAddress(address);
    SetLogicalAddress(address);
    UInt physicalAddress[4] = { ~static_cast<UInt>(0U), ~static_cast<UInt>(0U), ~static_cast<UInt>(0U), ~static_cast<UInt>(0U) };
    SetPhysicalAddress(physicalAddress);
}

    void DirectTextureImage::SetVideoMemoryHandle(Handle handle)
{
    m_videoMemoryHandle[ContextResourcePool::GetActive().GetIndex()] = handle;
}

void DirectTextureImage::SetVideoMemoryAddress(void* const address[4])
{
    const Int poolIndex = ContextResourcePool::GetActive().GetIndex();
    m_videoMemoryAddress[poolIndex][0] = address[0];
    m_videoMemoryAddress[poolIndex][1] = address[1];
    m_videoMemoryAddress[poolIndex][2] = address[2];
    m_videoMemoryAddress[poolIndex][3] = address[3];
}

void DirectTextureImage::GetVideoMemoryAddress(void* address[4]) const
{
    const Int poolIndex = ContextResourcePool::GetActive().GetIndex();
    address[0] = m_videoMemoryAddress[poolIndex][0];
    address[1] = m_videoMemoryAddress[poolIndex][1];
    address[2] = m_videoMemoryAddress[poolIndex][2];
    address[3] = m_videoMemoryAddress[poolIndex][3];
}


FEATSTD_RTTI_DEFINITION(DirectTextureImage, Base)
} // namespace Candera
