//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_GRAPHIC_PACKAGE_INTERFACE_H)
    #define CANDERAPLATFORM_GRAPHIC_PACKAGE_INTERFACE_H

#include <CanderaPlatform/OS/CanderaTypes.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageDescriptor.h>

namespace Candera {

/** @addtogroup CommonDevice
 *  @{
 */
    class Display;
    class GraphicDeviceUnit;
    class GpuPerformanceDataProvider;
    class RenderTarget2D;
    class ContextResourcePool;
    class ContextProvider3D;


     /**
      * @brief DevicePackageInterface is a static class that defines the main interface
      * of a device package. 
      * It provides means of creating device related
      * objects in a platform independent way.
      * 
      * This class is the only recommended means of creating Displays and
      * GraphicDeviceUnits. It also provides more advanced features like
      * creating 2D contexts and attaching default 3D context providers.
      */
    class DevicePackageInterface {
        public:
        
            //////////////////////////////////////////////////////////////////////////
            //   Display Manipulation
            //////////////////////////////////////////////////////////////////////////
            /**
             *  Creates a display with associated id.
             *  If the display id is unsupported or a display with this id already exists or the display 
             *  creation fails, 0 is returned.
             *  @param displayId    A hardware dependent display identifier starting with 0. 
             *                      The meaning of the identifiers are specific to the implementation. 
             *                      Please refer to the associated documentation for further details.
             *  @return             A pointer to a display or 0.
             */
            static Display* CreateDisplay(Int displayId);
            
            /**
             *  Any display created with CreateDisplay should be destroyed with this function
             *  @param display      Pointer to the object to be destroyed.
             */
            static void DestroyDisplay(Display* display);
            
            /**
             *  Provides a display object, previously created by CreateDisplay.
             *  @param displayId    Identifier used on creation.
             *  @return             A display object, 0 if it does not exist.
             */
            static Display* GetDisplay(Int displayId);
            
            //////////////////////////////////////////////////////////////////////////
            //   Graphic Device Unit Manipulation
            //////////////////////////////////////////////////////////////////////////
            
            /**
             *  Creates a new graphic device unit instance of the specified type.
             *  A common use case is when an application wants to have platform objects being created
             *  out of asset libraries where concrete types are known.
             *  @param type             A platform defined type. If a type is not supported, 0 is returned.
             *  @param configuration    A list of attribute and value pairs that are used as hints on 
             *                          how the new graphic device unit should be initialized. End of list has to be marked
             *                          with 0 (e.g. Int configuration[] = {GduWidthAttr, 800, GduHeightAttr, 600, GduAlphaSizeAttr, 8, 0};
             *  @return                 Pointer to created GraphicDeviceUnit according to the specified type.
             *  @see GraphicDeviceUnitAttributes
             */
            static GraphicDeviceUnit* CreateGraphicDeviceUnit(GraphicDeviceUnitTypeHandle type, const Int* configuration = 0);

            /**
             *  Applications are responsible for controlling the lifetime of platform objects. 
             *  Platform objects which have been created using CreateGraphicDeviceUnit need 
             *  to be destroyed using this method.
             *  @param object           GraphicDeviceUnit to be destroyed.
             */
            static void DestroyGraphicDeviceUnit(GraphicDeviceUnit* object);
            
            
            //////////////////////////////////////////////////////////////////////////
            //   Designer Support
            //////////////////////////////////////////////////////////////////////////
           
            /**
             *  Create a context handle for the specified RenderTarget2D, so that one can render
             *  2D context using RenderDevice2D. ContextHandle should be destroyed before destroying the
             *  render target. Should most probably be called when uploading the render target, so that other
             *  external contexts (for instance EGL context) is properly set.
             *  @param renderTarget     Render target object that will use the context.
             *  @return                 A context handle, 0 if association fails.
             */
            static ContextHandle2D CreateRenderContext2D(const RenderTarget2D* renderTarget);
            
            /**
             *  Destroy the context associated with the handle.
             *  @param context      A context handle created by CreateRenderContext2D.
             */
            static void DestroyRenderContext2D(ContextHandle2D context);
            
            /**
             *  Refresh context properties, based on the new render target properties.
             *  Should be called each time the render target modifies its properties, like width and height.
             *  @param context      A context handle created by CreateRenderContext2D.
             */
            static bool UpdateRenderContext2D(ContextHandle2D context);

            /**
             *  Create a default context provider in the given pool. This provides an interface
             *  for attaching default native handles to the context resource pool. Resources
             *  created within the pool may use these handles on creation. This mechanism
             *  allows creation of resources in an independent order. For instance frame buffer objects
             *  can be created within the pool without a display being available.
             *  Only one default context provider can be attached per pool. Farther calls on the same
             *  pool will fail before the ContextProvider is destroyed.
             *  If this feature is not supported the call will fail.
             *  The lifetime of the attached objects must be insured by the called of this function.
             *  DestroyDefaultContextProvider should be called before their destruction.
             *  @param pool     The context resource pool in which this context provider is created.
             *  @param display  A display handle for the context provider. Regularly an object of type EGLDisplay.
             *  @param config   A configuration object for the context provider. Regularly an object of type EGLConfig.
             *  @param surface  A surface handle for the context provider. Regularly an object of type EGLSurface.
             *  @param context  A context handle for the context provider. Regularly an object of type EGLContext.
             *  @return         A pointer to the created context provider, or 0 on failure.
             */
            static ContextProvider3D* CreateDefaultContextProvider(ContextResourcePool& pool, void* display, void* config, void* surface, void* context);
            
            /**
             *  Destroy a context provider.
             *  @param provider      An object created by CreateDefaultContextProvider.
             */
            static void DestroyDefaultContextProvider(const ContextProvider3D* provider);
            
            //////////////////////////////////////////////////////////////////////////
            //   Performance Data Support
            //////////////////////////////////////////////////////////////////////////
            
            /**
             *  Retrieve a provider for GPU performance data like GPU load.
             *  @return     A provider object, 0 if it does not exist.
             */
            static GpuPerformanceDataProvider* GetGpuPerformanceDataProvider();
    };

/** @}*/ //end of CommonDevice
}   //namespace Candera

#endif  // CANDERAPLATFORM_GRAPHIC_PACKAGE_INTERFACE_H

