//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined (CANDERAPLATFORM_DEVICE_PACKAGE_DESCRIPTOR_H)
    #define CANDERAPLATFORM_DEVICE_PACKAGE_DESCRIPTOR_H

#include <CanderaPlatform/OS/CanderaTypes.h>

namespace Candera
{
    class DeviceMetaInfoHost;
    namespace MetaInfo {
        class DeviceMetaInfo;
        typedef DeviceMetaInfo GraphicDeviceUnitMetaInfo;
    }
    typedef Handle GraphicDeviceUnitTypeHandle;

    /**
     *  Describes the basic functionality of the platform.
     */
    struct PlatformDescription {
        bool is2DSupported;                 ///<    Defines whether the platform supports 2D rendering or not.
        bool is3DSupported;                 ///<    Defines whether the platform supports 3D rendering or not.
        bool is2DBitmapVerticallyFlipped;   ///<    Defines if 2D Bitmaps are vertically flipped (lower left corner has coordinates 0/0) of not (upper left corner has coordinates 0/0).
        bool is3DBitmapVerticallyFlipped;   ///<    Defines if 3D Bitmaps are vertically flipped (lower left corner has coordinates 0/0) of not (upper left corner has coordinates 0/0).
        bool isWindowManagerUsed;           ///<    Defines whether the platform uses a Window Manager.
    };

/**
 *  DevicePackageDescriptor
 */
class DevicePackageDescriptor {
    public:
        enum UnitCategory {
            DisplayTarget3D,            ///< 3D display target.
            DisplayTarget2D,            ///< 2D display target.
            Mixed2D3DDisplayTarget,     ///< 2D & 3D mixed display target.
            OffscreenTarget3D,          ///< 3D offscreen target.
            OffscreenTarget2D,          ///< 2D offscreen target.
            Mixed2D3DOffscreenTarget,   ///< 2D & 3D mixed offscreen target.
            OffscreenTarget2Dto3D,
            OffscreenTarget3Dto2D,
            ExternalSource,
            CustomObject
        };

        enum DisplayAdapter {
            DefaultAdapter      ///< Default display adapter.
        };

        enum DeviceType {
            TargetDevice,       ///< Target device.
            SimulationDevice    ///< Simulation device.
        };

        struct RenderDevice3DCapabilities {
            struct ViewportSize {
                UInt32 width;
                UInt32 height;
            };

            UInt32 m_maxTextureSize;                ///< Largest texture size that the platform render device can handle.
            UInt8 m_maxCombinedTextureImageUnits;   ///< Maximum number of TextureImages that can be used by the vertex shader and the fragment processor combined.
            UInt32 m_maxCubeMapTextureSize;         ///< Largest cube map texture size that the platform render device can handle.
            UInt16 m_maxFragmentUniformVectors;     ///< Maximum number of vectors that can be held in fragment shader uniforms.
            UInt32 m_maxRenderBufferSize;           ///< Largest render-buffer size in bytes that the platform render device can handle.
            UInt8 m_maxTextureImageUnits;           ///< Maximum number of TextureImages that can be used by the fragment shader.
            UInt8 m_maxVaryingVectors;              ///< Maximum number of vectors available for interpolating varying variables used by vertex and fragment shaders.
            UInt8 m_maxVertexAttributes;            ///< Maximum number of generic vertex attributes accessible to a vertex shader.
            UInt8 m_maxVertexTextureImageUnits;     ///< Maximum number of TextureImages that can be used by the vertex shader.
            UInt16 m_maxVertexUniformVectors;       ///< Maximum number of vectors that can be held in vertex shader uniforms.
            ViewportSize m_maxViewportSize;         ///< Maximum supported width and height of the viewport.
            UInt32 m_maxElementIndex;               ///< Maximum index value for an indexed vertex buffer.
            const Char* m_rendererAPI;              ///< String containing the API name (e.g. "OpenGL ES 2.0")
            bool m_isDxtCompressionSupported;       ///< Flag that specifies if DXT compressed textures can be handled.
        };

        /**
         *  Provides the name of this device package.
         *  @return     A string containing the name of the package.
         */
        static const Char* GetGraphicPackageName();

        /**
         *  Provides the name of this device package instance.
         *  @return     A string containing the name of the package.
         */
        static const Char* GetGraphicPackageInstanceName();

        /**
         *  Provides available graphic device unit types associated with a display.
         *  Off-screen graphic device units are available to all displays.
         *  @param category (input)     Category of the requested graphic device unit.
         *  @param display (input)      Display identifier.
         *  @param count (output)       The number of available graphic device units with this category on the specified display.
         *  @return                     An array of graphic device unit type handles.
         */
        static const GraphicDeviceUnitTypeHandle* GetUnitTypes(UnitCategory category, Int display, Int& count);

        /**
         *  Determines the category of the graphic device unit type.
         *  @param unitType     Graphic device unit type.
         *  @return             The category of the graphic device unit type.
         */
        static UnitCategory GetUnitCategory(GraphicDeviceUnitTypeHandle unitType);

        /**
         *  Provides meta info associated with a graphic device unit.
         *  @param objectType   Type handle for the graphic device unit.
         *  @return             A meta information object.
         */
        static const Candera::MetaInfo::GraphicDeviceUnitMetaInfo* GetMetaInformation(GraphicDeviceUnitTypeHandle objectType);

        /**
         *  Provides the name of the shader compiler needed for this device package.
         *  This is the shader compiler for shaders used on target.
         *  @return     String containing the name.
         */
        static const Char* GetGraphicPackageShaderCompiler();

        /**
         *  Provides the name of the shader compiler needed for this device package instance.
         *  This is the shader compiler for shaders used in the current context.
         *  @return     String containing the name.
         */
        static const Char* GetGraphicPackageInstanceShaderCompiler();

        /**
         *  Provides a semi-colon separated list of available fractional number formats.
         *  (e.g. "float; int16:16")
         *  @return     String containing the the list.
         */
        static const Char* GetAvailableFractionalNumberFormats();

        /**
         *  Provides the number of available displays for this device package.
         *  @return     Integer containing the number of available displays.
         */
        static Int GetNumberOfDisplays();

        /**
         *  Provides meta info associated with a display.
         *  @param displayId    Identifier for the display.
         *  @return             A meta information object.
         */
        static const Candera::MetaInfo::DeviceMetaInfo* GetDisplayMetaInformation(Int displayId);

        /**
         *  Provides the context resource pool index associated with a display.
         *  @param displayId    Identifier for the display.
         *  @return             A context resource pool index, -1 if the display
         *                      doesn't associate with context resource pools.
         */
        static Int GetDisplayContextResourcePoolIndex(Int displayId);

        /**
         *  Provides meta info associated with a the current device package.
         *  @return     A meta information object.
         */
        static const Candera::MetaInfo::DeviceMetaInfo* GetPackageMetaInformation();

        /**
         *  Provides an interface to access the properties of the current device package.
         *  @return     A pointer to a meta information host object.
         */
        static Candera::DeviceMetaInfoHost* GetMetaInformationHost();

        /**
         *  Provides a platform description structure for this device package.
         *  @return     A platform description object.
         */
        static const PlatformDescription& GetPlatformDescription();

        /**
         *  Provides RenderDevice3D capabilities.
         *
         *  @param displayAdapter   Adapter ID.
         *  @param deviceType       Device type (target or simulation).
         *  @param capabilities     RenderDevice3DCapabilities of RenderDevice3D if 3D rendering is available.
         *  @return                 true on success, false otherwise.
         */
        static bool GetRenderDevice3DCapabilities(DisplayAdapter displayAdapter, DeviceType deviceType, const RenderDevice3DCapabilities*& capabilities);
    private:
        //static interface. no instance creation allowed.
        DevicePackageDescriptor();
        ~DevicePackageDescriptor();

        /**
         * Queries the device's capabilities from RenderDevice. Stores
         * the queried capabilities in a static RenderDevice3DCapabilities struct,
         * and returns its const pointer.
         * @return      true if querying capabilities succeeded.
         */
        static bool GetCurrentRenderDevice3DCapabilities(const RenderDevice3DCapabilities*& capabilities);
    };

}   // namespace Candera

#endif  // CANDERAPLATFORM_DEVICE_PACKAGE_DESCRIPTOR_H

