//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_CONTEXT_PROVIDER_3D_H)
    #define CANDERAPLATFORM_CONTEXT_PROVIDER_3D_H

#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>
#include <Candera/System/Container/LinkedList.h>
#include <FeatStd/Macros.h>

namespace Candera {

/** @addtogroup CommonDevice
 *  @{
 */

// Forward declaration.
class ContextResourcePool;
struct RenderStateCache;

/**
 *  @brief A ContextProvider3D is a render target that owns a distinctive context to store render states.
 *  The context of a ContextProvider3D object is utilized by ContextResourcePool to share its device objects like
 *  Shaders, TextureImages or VertextBuffers with other contexts.
 */
class ContextProvider3D : public RenderTarget3D {
    public:
        /**
         * Constructs a ContextProvider3D object.
         */
        ContextProvider3D() : RenderTarget3D(0), m_contextResourcePool(0), m_renderStates(0) {}
        /**
         * Destructs a ContextProvider3D object.
         */
        virtual ~ContextProvider3D();

        /**
         * Set context resource pool for this RenderTarget3D object.
         * Unsubscribes from current resource pool if set, and subscribes to 
         * given resource pool if not null.
         * This function is invoked automatically when a Graphic Device Unit is 
         * uploaded to a display.
         * @param resourcePool  Pointer to the resource pool to subscribe to.
         * @return              true if the ContextProvider was set successfully,
         *                      false otherwise.
         */
        bool SetContextResourcePool(ContextResourcePool* resourcePool);

        /**
         * Retrieve the context resource pool to which this object is subscribed.
         * @return  The associated context resource pool.
         */
        virtual const ContextResourcePool* GetContextResourcePool(
            ) const override final { return m_contextResourcePool; }
        using RenderTarget3D::GetContextResourcePool;

        /**
         * Get the shared context handle from the context resource pool.
         * @return  The native context handle.
         */
        void* GetSharedContextHandle() const;

    protected:
        /**
         * Make the associated resource pool active. This should be called by RenderTarget::Activate,
         * such that Device objects determine the correct context resource pool to upload to.
         */
        void ActivateContextResourcePool();

        /**
         * Activates the cached render state of this ContextProvider3D.
         * @return  true if cache was set to RenderDevice, false otherwise.
         */
        void ActivateRenderStateCache();
        /**
         * Creates a render state cache object
         * @return  True if render state cache object was created successfully.
         */
        bool CreateRenderStateCache();
        /**
         * Destroys a render state cache object.
         */
        void DestroyRenderStateCache();

        /**
         * Let the RenderDevice know that this context will be destroyed.
         */
        void OnContextUnload() const;

    private:
        friend class ContextResourcePool;

        // Context resource pool this ContextProvider3D object is associated to.
        ContextResourcePool* m_contextResourcePool;
        RenderStateCache* m_renderStates;

        // ContextProvider3D points to next ContextProvider3D object in order to be linkable for ContextResourcePool.
        Internal::LinkedListNode<ContextProvider3D> m_contextProviderNode;
};

/** @}*/ //end of CommonDevice

}   // namespace Candera

#endif  // CANDERAPLATFORM_CONTEXT_PROVIDER_3D_H
