//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GLOBALIZATION_GLOBALIZATIONCULTUREMANAGER_H)
#define CANDERA_GLOBALIZATION_GLOBALIZATIONCULTUREMANAGER_H

#include <Candera/System/GlobalizationBase/CultureManager.h>
#include <Candera/System/Container/SingleLinkedList.h>
#include <CanderaGlobalization/Localizer.h>

namespace Candera { namespace Globalization {
/// @addtogroup CanderaGlobalizationCulture
/// @{

/**
    *  API for accessing and changing the current culture as well as the list of available cultures.
    *  The currently set culture defines, together with other settings, the language and is therefore
    *  responsible for text translation.
    */
class GlobalizationCultureManager : public CultureManager {
    public:

        FEATSTD_LINT_NEXT_EXPRESSION(1511, "disambiguity with CultureManager::GetInstance remains in place to preserve interface compatibility.")
        static GlobalizationCultureManager& GetInstance();

        /**
            *  Sets the current culture.
            *  @param  locale      The locale string which implicitly defines the culture.
            *  @return             true, if the culture has been set successfully.
            */
        virtual bool SetCurrentCulture(const Char* locale);

        /**
            *  Sets the current culture.
            *  @param  culture     The culture to use.
            *  @return             true, if the culture has been set successfully.
            */
        FEATSTD_LINT_NEXT_EXPRESSION(1411, "disambiguity with CultureManager::SetCurrentCulture remains in place to preserve interface compatibility.")
        virtual bool SetCurrentCulture(Culture::SharedPointer culture);

        /**
            *  Retrieves the current culture.
            *  @return     The currently set culture.
            */
        virtual Culture::SharedPointer GetCurrentCulture() const;

        /**
            *  Retrieves the amount of cultures which have been registered to the CultureManager.
            *  @return     The count of currently available cultures.
            */
        virtual UInt16 GetCultureCount() const;

        /**
            *  Retrieves a specific culture within the list of all registered cultures.
            *  @param  index       The index in the list of available cultures.
            *  @return             The culture with the specified index, 0 if this culture is not available.
            */
        virtual Culture::SharedPointer GetCulture(UInt16 index) const;

        /**
            *  Retrieves a specific culture within the list of all registered cultures.
            *  @param  locale      The locale string which implicitly defines the culture.
            *  @return             The culture for locale, 0 if this culture is not available.
            */
        virtual Culture::SharedPointer GetCulture(const Char* locale) const;

        /**
            *  Adds a culture to the list of available cultures.
            *  @param  culture     The culture to be added.
            */
        void AddCulture(Culture::SharedPointer culture);

        /**
            *  Removes a culture from the list of available cultures.
            *  @param  index       The index in the list of available cultures.
            *  @return             Pointer to the successfully removed culture, otherwise 0.
            */
        Culture::SharedPointer RemoveCulture(UInt16 index);

        /**
            *  Removes a culture from the list of available cultures.
            *  @param  locale      The locale string which implicitly defines the culture to be removed.
            *  @return             Pointer to the successfully removed culture, otherwise 0.
            */
        Culture::SharedPointer RemoveCulture(const Char* locale);

        /**
            *  Removes a culture from the list of available cultures.
            *  @param  culture      The culture to removed.
            *  @return              true, if removed successfully, false otherwise.
            */
        bool RemoveCulture(Culture::SharedPointer culture);

        /**
            *  Removes all cultures from this culture manager.
            */
        void RemoveAllCultures();

        /**
            *  Registers a localizer instance (e.g. from asset loader) to the CultureManager.
            *  As long as there is no localizer registered to the CultureMananger, no text translation will occur.
            *  It is only allowed to register one localizer at the same time. To unregister a localizer, SetLocalizer(0) has to be called.
            *  @param  localizer   The localizer to be set.
            */
        void SetLocalizer(const Localizer::SharedPointer& localizer);

        /**
            *  Retrieves the currently set localizer.
            *  @return     The currently set localizer, 0 if no localizer has been set.
            */
        const Localizer::SharedPointer& GetLocalizer() const { return m_localizer; }

        /**
         *  Destructor
         */
        virtual ~GlobalizationCultureManager();

    protected:
        virtual void NotifyCultureChangeListeners(Culture::SharedPointer culture);

    private:
        typedef Candera::Internal::SingleLinkedList<Culture::SharedPointer> CultureList;

        Culture::SharedPointer m_currentCulture;
        Localizer::SharedPointer m_localizer;
        CultureList m_cultureList;

        GlobalizationCultureManager();

        friend class CultureManager;
};
/// @}
}}
#endif  // CANDERA_GLOBALIZATION_GLOBALIZATIONCULTUREMANAGER_H
