//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlobalizationCultureManager.h"
#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Util/StaticObject.h>
#include <CanderaPlatform/OS/StringPlatform.h>
#include <CanderaGlobalization/LocalizableStringDataFactory.h>
#include <FeatStd/Util/NumericUtil.h>
#if defined(FEATSTD_THREADSAFETY_ENABLED)
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace Candera { namespace Globalization {
using namespace Internal;

/******************************************************************************
 *  GetInstance
 ******************************************************************************/
CultureManager& CultureManager::GetInstance()
{
    return GlobalizationCultureManager::GetInstance();
}

GlobalizationCultureManager& GlobalizationCultureManager::GetInstance()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(GlobalizationCultureManager, s_globalizationCultureManager);
    return s_globalizationCultureManager;
}

static GlobalizationCultureManager& s_forceInitInstance = GlobalizationCultureManager::GetInstance();

/******************************************************************************
 *  Constructor
 ******************************************************************************/
GlobalizationCultureManager::GlobalizationCultureManager() :
    m_currentCulture(),
    m_localizer(),
    m_cultureList()
{
    //Initializes LocalizableStringDataFactory.
    static_cast<void>(LocalizableStringDataFactory::GetInstance());
}

/******************************************************************************
 *  Destructor
 ******************************************************************************/
GlobalizationCultureManager::~GlobalizationCultureManager()
{
}

/******************************************************************************
 *  GetCurrentCulture
 ******************************************************************************/
Culture::SharedPointer GlobalizationCultureManager::GetCurrentCulture() const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    return (m_currentCulture != 0) ? m_currentCulture : GetDefaultCulture();
}

/******************************************************************************
 *  SetCurrentCulture
 ******************************************************************************/
bool GlobalizationCultureManager::SetCurrentCulture(const Char* locale)
{
    return SetCurrentCulture(GetCulture(locale));
}

/******************************************************************************
 *  SetCurrentCulture
 ******************************************************************************/
bool GlobalizationCultureManager::SetCurrentCulture(Culture::SharedPointer culture)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    if ((culture != 0) && (culture != m_currentCulture)) {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        lock.Release();
#endif
        NotifyCultureChangeListeners(culture);
#ifdef FEATSTD_THREADSAFETY_ENABLED
        lock.Obtain();
#endif
        m_currentCulture = culture;
        return true;
    }

    return false;
}

/******************************************************************************
 *  AddCulture
 ******************************************************************************/
void GlobalizationCultureManager::AddCulture(Culture::SharedPointer culture)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    if (culture != 0) {
        static_cast<void>(m_cultureList.Append(culture));
    }
}

/******************************************************************************
 *  RemoveAllCultures
 ******************************************************************************/
void GlobalizationCultureManager::RemoveAllCultures()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    m_cultureList.Clear();
}

/******************************************************************************
 *  RemoveCulture
 ******************************************************************************/
Culture::SharedPointer GlobalizationCultureManager::RemoveCulture(UInt16 index)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    Culture::SharedPointer culture = GetCulture(index);
    return RemoveCulture(culture) ? culture : Culture::SharedPointer();
}

/******************************************************************************
 *  RemoveCulture
 ******************************************************************************/
bool GlobalizationCultureManager::RemoveCulture(Culture::SharedPointer culture)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    if (m_currentCulture == culture) {
        m_currentCulture = Culture::SharedPointer();
    }

    return m_cultureList.Remove(culture);
}

/******************************************************************************
 *  RemoveCulture
 ******************************************************************************/
Culture::SharedPointer GlobalizationCultureManager::RemoveCulture(const Char* locale)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    for (CultureList::Iterator it = m_cultureList.Begin(); it != m_cultureList.End(); ++it) {
        if (StringPlatform::CompareStrings((*it)->GetLocale(), locale) == 0) {
            Culture::SharedPointer culture = *it;
            return RemoveCulture(culture) ? culture : Culture::SharedPointer();
        }
    }

    return Culture::SharedPointer();
}

/******************************************************************************
 *  GetCulture
 ******************************************************************************/
Culture::SharedPointer GlobalizationCultureManager::GetCulture(UInt16 index) const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    if (index < m_cultureList.GetSize()) {
        CultureList::ConstIterator it = m_cultureList.Begin();
        for (Int i = 0; i < index; i++) {
            ++it;
        }
        return (*it);
    }

    return Culture::SharedPointer();
}

/******************************************************************************
 *  GetCulture
 ******************************************************************************/
Culture::SharedPointer GlobalizationCultureManager::GetCulture(const Char* locale) const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    for (CultureList::ConstIterator it = m_cultureList.Begin(); it != m_cultureList.End(); ++it) {
        if (StringPlatform::CompareStrings((*it)->GetLocale(), locale) == 0) {
            return (*it);
        }
    }

    return Culture::SharedPointer();
}

/******************************************************************************
 *  SetLocalizer
 ******************************************************************************/
void GlobalizationCultureManager::SetLocalizer(const Localizer::SharedPointer& localizer)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    // avoid to overwrite an existing localizer
    FEATSTD_DEBUG_ASSERT(m_localizer == 0 || localizer == 0);
    m_localizer = localizer;
}

/******************************************************************************
 *  NotifyCultureChangeListeners
 ******************************************************************************/
void GlobalizationCultureManager::NotifyCultureChangeListeners(Culture::SharedPointer culture)
{
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

        if (!m_localizer.PointsToNull()) {
            m_localizer->SetCurrentCulture(culture);
        }
    }

    CultureManager::NotifyCultureChangeListeners(culture);
}

/******************************************************************************
 *  GetCultureCount
 ******************************************************************************/
UInt16 GlobalizationCultureManager::GetCultureCount() const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    UInt32 size = FeatStd::Internal::NumericConversion<UInt32, SizeType>(m_cultureList.GetSize());
    FEATSTD_DEBUG_ASSERT(size < 0xFFFF);
    return static_cast<UInt16>(size);
}
}}
