//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "DefaultLocalizer.h"
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>
#include <FeatStd/Util/StaticObject.h>
#if defined(FEATSTD_THREADSAFETY_ENABLED)
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace Candera { namespace Globalization { namespace Internal {
/******************************************************************************
 *  Constructor
 ******************************************************************************/
DefaultLocalizer::DefaultLocalizer() :
    m_assetProvider(0)
{
}

/******************************************************************************
 *  Destructor
 ******************************************************************************/
DefaultLocalizer::~DefaultLocalizer()
{
}

/******************************************************************************
 *  Create
 ******************************************************************************/
DefaultLocalizer::SharedPointer DefaultLocalizer::Create()
{
    return DefaultLocalizer::SharedPointer(FEATSTD_NEW(DefaultLocalizer));
}

/******************************************************************************
 *  GetInstance
 ******************************************************************************/
DefaultLocalizer& DefaultLocalizer::GetInstance()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(DefaultLocalizer::SharedPointer, s_defaultLocalizer, Create());
    return s_defaultLocalizer.GetSharedInstance();
}

static DefaultLocalizer& s_forceInitInstance = DefaultLocalizer::GetInstance();

#ifdef FEATSTD_THREADSAFETY_ENABLED
static FeatStd::Internal::CriticalSection& GetCriticalSection()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, cs);
    return cs;
}

static FeatStd::Internal::CriticalSection& s_forceInitCriticalSection = GetCriticalSection();
#endif

/******************************************************************************
 *  SetCurrentCulture
 ******************************************************************************/
void DefaultLocalizer::SetCurrentCulture(Culture::SharedPointer culture)
{
    if (culture == 0) {
        return;
    }

#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    if (m_assetProvider != 0) {
        const Char* locale = culture->GetLocale();
        m_languagePack = m_assetProvider->GetLanguagePack(locale);
    }
}

/******************************************************************************
 *  GetTextBaseString
 ******************************************************************************/
const TChar* DefaultLocalizer::GetTextBaseString(Localizable::Id id) const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    return (m_languagePack != 0) ? m_languagePack->GetBaseString(id) : 0;
}

/******************************************************************************
 *  GetTextType
 ******************************************************************************/
DefaultLocalizer::Base::TextType DefaultLocalizer::GetTextType(Localizable::Id id) const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    return (m_languagePack != 0) ? m_languagePack->GetTextType(id) : _undefined;
}
}}}   // namespace Candera::Globalization::Internal
