//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GLOBALIZATION_PARAMETERIZEDLOCALIZABLETEXT_H)
#define CANDERA_GLOBALIZATION_PARAMETERIZEDLOCALIZABLETEXT_H

#include <Candera/Environment.h>
#include <CanderaGlobalization/DataBinding/ConditionalFormatPattern.h>
#include <CanderaGlobalization/DataBinding/Formatter.h>
#include <CanderaGlobalization/LocalizableText.h>
#include <Courier/DataBinding/DataItemType.h>
#include <Courier/DataBinding/ModelBindingTargetHost.h>
#include <FeatStd/Util/PointerUtil.h>

namespace Courier {
    class DataItemValue;

    namespace Internal {
        template <typename T> class BindableParameterCapability;
        class AsyncTextParameterBinding;
        class Binding;
    }
}

namespace Candera { namespace Globalization { namespace Internal {
/// @addtogroup CanderaGlobalizationLocalizable
/// @{
/**
    * Represents a text parameter.
    */
class ParameterBase {
public:
    Courier::DataItemValue& GetValue() { return m_value; }
    void SetValue(const Courier::DataItemValue& value) { m_value.SetValue(value); }
protected:
    Courier::DataItemValue m_value;
};

template <typename T> class Parameter : public ParameterBase {
public:
    Parameter() {
        InitializeData();
        m_value = Courier::DataItemValue(Courier::DataItemType<T>::Ident(), &m_data, true);
    }
private:
    T m_data;

    void InitializeData() {}
};

typedef Parameter<FeatStd::Int32>  Int32Parameter;
typedef Parameter<FeatStd::Float>  FloatParameter;
typedef Parameter<bool>   BoolParameter;
typedef Parameter<FeatStd::String> StringParameter;

/**
    *  Supports internationalization for texts containing parameterized, conditional format patterns (e.g. "Track {0} of {1}").
    */
class ParameterizedLocalizableText : public LocalizableTextBase, public Courier::Internal::ModelBindingTargetHost {
    FEATSTD_TYPEDEF_BASE(LocalizableTextBase);

    public:
        ParameterizedLocalizableText(Id id);
        virtual ~ParameterizedLocalizableText();

        bool SetValue(UInt8 index, const Courier::DataItemValue& value);

        bool Enlist(Courier::Internal::Binding* binding);
        bool Unlist(Courier::Internal::Binding* binding);

        /**
            *  Initializes the parameterized text.
            *  @param  hdl          Handle to the binary data representation.
            *  @param updateBinding Whether data binding shall be updated.
            */
        bool Init(Handle hdl, bool updateBinding); // TODO: set private

    protected:

        virtual void OnIdChanged(Id id);
        virtual void OnCultureChanged(const Culture& culture);
        virtual void ResolveString() const;

    private:
        friend class FormatPattern;
        friend class Operand;

        struct Header {
            UInt16 m_maxStingSize;
            UInt16 m_paramOffset;
            UInt16 m_conditionOffset;
            UInt16 m_padding;
        };

        mutable Handle m_hdl;

        TChar* m_buffer;
        UInt8 m_parameterCount;       // cached value to avoid multiple calls to GetParamCount().
        ParameterBase** m_parameters;
        Courier::Internal::Binding* m_BindingsHead;

        void UpdateBaseString(Id id);

        const Courier::DataItemValue& GetParameterValue(UInt8 index) const;

        const Header& GetHeader() const {
            return *FeatStd::Internal::PointerToPointer<const Header*>(m_hdl);
        }

        Handle GetConditionalFormatPatternHandle() const {
            return GetHandleForOffset(GetHeader().m_conditionOffset);
        }

        Handle GetParametersHandle() const {
            return GetHandleForOffset(GetHeader().m_paramOffset);
        }

        UInt16 GetMaxStringSize() const { return GetHeader().m_maxStingSize; }

        UInt8 GetParamCount() const { Parameters parameter(GetParametersHandle()); return parameter.GetCount(); }

        UInt16 Format(UInt8 paramIndex, TChar* dst) const;
        Formatter* GetFormatter(UInt8 paramIndex) const;

        bool SetupParameters();
        void TeardownParameters();
        ParameterBase* CreateParameter(FeatStd::UInt8 index);
        bool SetupBindings();
        void TeardownBindings();
        bool SetupBuffer();
        void TeardownBuffer();

        template <typename T> friend class Courier::Internal::BindableParameterCapability;
        friend class Courier::Internal::AsyncTextParameterBinding;
};
/// @}
}}}
#endif  // CANDERA_GLOBALIZATION_PARAMETERIZEDLOCALIZABLETEXT_H
