//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ParameterizedLocalizableText.h"

#include <FeatStd/Diagnostics/Debug.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>
#include <CanderaGlobalization/GlobalizationCultureManager.h>
#include <CanderaGlobalization/Localizer.h>
#include <Courier/DataBinding/AsyncTextParameterBindingFactory.h>
#include <Courier/DataBinding/Binding.h>
#include <Courier/DataBinding/DataBinding.h>
#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

using namespace Courier;
using namespace Courier::Internal;
using namespace Courier::Internal::DataBinding;

namespace Courier {
    namespace Internal {
        typedef ObjectListFunc<Binding, BindingTargetHostListAccessor> TextBindingList;
        typedef ObjectListIterator<Binding, BindingTargetHostListAccessor> TextBindingListIt;
    }
}

namespace Candera { namespace Globalization { namespace Internal {
template <> void Parameter<Int32>::InitializeData() {
    m_data = 0;
}

template <> void Parameter<Float>::InitializeData() {
    m_data = 0.0f;
}

template <> void Parameter<bool>::InitializeData() {
    m_data = false;
}

template <> void Parameter<FeatStd::String>::InitializeData() {
    m_data = FeatStd::String();
}

/******************************************************************************
 *  Constructor
 ******************************************************************************/
ParameterizedLocalizableText::ParameterizedLocalizableText(Id id) :
    Base(id),
    m_hdl(0),
    m_buffer(0),
    m_parameterCount(0),
    m_parameters(0),
    m_BindingsHead(0)
{
    UpdateBaseString(id);
}

/******************************************************************************
 *  Destructor
 ******************************************************************************/
ParameterizedLocalizableText::~ParameterizedLocalizableText()
{
    TeardownBindings();
    TeardownParameters();
    TeardownBuffer();
    m_hdl = 0;
}

/******************************************************************************
 *  Init
 ******************************************************************************/
bool ParameterizedLocalizableText::Init(Handle hdl, bool updateBindings)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    m_hdl = hdl;

    TeardownBuffer();
    bool ok = SetupBuffer();

    if (updateBindings) {
        TeardownParameters();
        ok = ok && SetupParameters();

        TeardownBindings();
        ok = ok && SetupBindings();
    }

    return ok;
}

/******************************************************************************
 *  SetupParameters
 ******************************************************************************/
bool ParameterizedLocalizableText::SetupParameters()
{
    m_parameterCount = GetParamCount();

    m_parameters = CANDERA_NEW_ARRAY(ParameterBase*, m_parameterCount);
    bool ok = m_parameters != 0;
    if (ok){
        for (UInt8 i = 0; i < m_parameterCount; i++) {
            m_parameters[i] = CreateParameter(i);
            ok = ok && (m_parameters[i] != 0);
        }

        if (!ok) {
            TeardownParameters();
        }
    }
    return ok;
}

/******************************************************************************
 *  TeardownParameters
 ******************************************************************************/
void ParameterizedLocalizableText::TeardownParameters() {
    if (m_parameters != 0) {
        for (UInt8 i = 0; i < m_parameterCount; i++) {
            if (m_parameters[i] != 0) {
                CANDERA_DELETE(m_parameters[i]);
                m_parameters[i] = 0;
            }
        }
        CANDERA_DELETE_ARRAY(m_parameters);
        m_parameters = 0;
    }
}

/******************************************************************************
 *  CreateParameter
 ******************************************************************************/
ParameterBase* ParameterizedLocalizableText::CreateParameter(UInt8 index)
{
    Parameters parameters(GetParametersHandle());
    switch (parameters.GetType(index)) {
        case Int32ValueType:
            return new Int32Parameter();
        case FloatValueType:
            return new FloatParameter();
        case BooleanValueType:
            return new BoolParameter();
        case StringValueType:
            return new StringParameter();
        default:
            return 0;
    }
}

/******************************************************************************
 *  SetValue
 ******************************************************************************/
bool ParameterizedLocalizableText::SetValue(UInt8 index, const DataItemValue& value)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    if (index < m_parameterCount) {
        m_parameters[index]->SetValue(value);
        m_cachedString = 0;
        return true;
    }

    return false;
}

/******************************************************************************
 *  OnIdChanged
 ******************************************************************************/
void ParameterizedLocalizableText::OnIdChanged(Id id)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    UpdateBaseString(id);
    Base::OnIdChanged(id);
}

/******************************************************************************
 *  OnCultureChanged
 ******************************************************************************/
void ParameterizedLocalizableText::OnCultureChanged(const Culture& culture)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    Localizer::SharedPointer localizer = GlobalizationCultureManager::GetInstance().GetLocalizer();
    if (!localizer.PointsToNull()) {
        m_hdl = localizer->GetTextBaseString(GetId()); // the handle to the base string
    }                                                  // has been potentially updated

    Base::OnCultureChanged(culture);
}

/******************************************************************************
 *  ResolveString
 ******************************************************************************/
void ParameterizedLocalizableText::ResolveString() const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    ConditionalFormatPattern formatPattern(GetConditionalFormatPatternHandle(), *this);
    formatPattern.ResolveConditions();
    m_cachedString = m_buffer;
}

/******************************************************************************
 *  Format
 ******************************************************************************/
UInt16 ParameterizedLocalizableText::Format(UInt8 paramIndex, TChar* dst) const
{
    UInt16 len = 0;
    if (paramIndex < m_parameterCount) {
        Formatter* formatter = GetFormatter(paramIndex);
        if (formatter != 0) {
            const DataItemValue& value = GetParameterValue(paramIndex);
            len = formatter->Format(value, dst, static_cast<UInt16>((GetMaxStringSize() - (dst - m_buffer)) - 1));
        }
    }

    return len;
}

/******************************************************************************
 *  GetFormatter
 ******************************************************************************/
Formatter* ParameterizedLocalizableText::GetFormatter(UInt8 paramIndex) const
{
    FEATSTD_DEBUG_ASSERT(paramIndex < m_parameterCount);

    Parameters params(GetParametersHandle());

    switch (params.GetType(paramIndex)) {
        case Int32ValueType: {
            static Int32Formatter formatter;
            formatter.SetFormatString(params.GetFormatString(paramIndex));
            return &formatter;
        }

        case FloatValueType: {
            static FloatFormatter formatter;
            formatter.SetFormatString(params.GetFormatString(paramIndex));
            return &formatter;
        }

        case StringValueType: {
            static StringFormatter formatter;
            return &formatter;
        }

        case BooleanValueType: {
            static BoolFormatter formatter;
            const TChar* str = reinterpret_cast<const TChar*>(params.GetFormatString(paramIndex));
            formatter.SetTrueString(str);
            while (*str != '\0') {
                str++;
            }
            formatter.SetFalseString(++str);
            return &formatter;
        }

        default:
            FEATSTD_DEBUG_FAIL();
            break;
    }

    return 0;
}

/******************************************************************************
 *  void
 ******************************************************************************/
void ParameterizedLocalizableText::UpdateBaseString(Id id)
{
    Localizer::SharedPointer localizer = GlobalizationCultureManager::GetInstance().GetLocalizer();
    if ((!localizer.PointsToNull()) && (id != 0)) {
        FEATSTD_DEBUG_ASSERT(localizer->GetTextType(id) == Localizer::ParameterizedText);
        Init(reinterpret_cast<Handle>(localizer->GetTextBaseString(id)), true);
    }
}

/******************************************************************************
 *  GetValue
 ******************************************************************************/
const DataItemValue& ParameterizedLocalizableText::GetParameterValue(UInt8 index) const
{
    FEATSTD_DEBUG_ASSERT(index < m_parameterCount);
    return m_parameters[index]->GetValue();
}

/******************************************************************************
 *  Enlist
 ******************************************************************************/
bool ParameterizedLocalizableText::Enlist(Binding* binding)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    return TextBindingList::Add(m_BindingsHead, binding);
}

/******************************************************************************
 *  Unlist
 ******************************************************************************/
bool ParameterizedLocalizableText::Unlist(Binding* binding)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    return TextBindingList::Remove(m_BindingsHead, binding);
}

/******************************************************************************
 *  SetupBindings
 ******************************************************************************/
bool ParameterizedLocalizableText::SetupBindings()
{
    const Parameters parameters(GetParametersHandle());
    bool ok = true;
    for (UInt8 i = 0; i < parameters.GetCount(); i++) {
        UInt32 hashValue = parameters.GetDataItemHash(i);
        const DataItemHierarchyNode *node = DataItemAccessor::Locate(hashValue);
        ok = ok && node != 0;
        ok = ok && AsyncTextParameterBindingFactory::GetInstance().CreateBinding(node->mItemKey, i, this);
    }
    return ok;
}

/******************************************************************************
 *  TeardownBindings
 ******************************************************************************/
void ParameterizedLocalizableText::TeardownBindings()
{
    TextBindingListIt iterator(m_BindingsHead);
    while (*iterator != 0) {
        Binding *binding = *iterator;
        ++iterator;
        binding->Destroy();
    }
    m_BindingsHead = 0;
}

/******************************************************************************
 *  SetupBuffer
 ******************************************************************************/
bool ParameterizedLocalizableText::SetupBuffer()
{
    FEATSTD_DEBUG_ASSERT(m_buffer == 0);
    m_buffer = CANDERA_NEW_ARRAY(TChar, GetMaxStringSize() + 1);
    return m_buffer != 0;
}

/******************************************************************************
 *  TeardownBuffer
 ******************************************************************************/
void ParameterizedLocalizableText::TeardownBuffer()
{
    if (m_buffer != 0) {
        CANDERA_DELETE_ARRAY(m_buffer);
        m_buffer = 0;
    }
}
}}}   //namespace Candera::Globalization::Internal
