//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ConditionalFormatPattern.h"

#include <CanderaGlobalization/DataBinding/ParameterizedLocalizableText.h>

#include <FeatStd/Diagnostics/Debug.h>

#include <FeatStd/Util/TextEncoding.h>
#include <FeatStd/Util/String.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

using FeatStd::String;
using FeatStd::Internal::TextEncoding;

using Courier::DataItemValue;
using Courier::DataItemType;

namespace Candera { namespace Globalization { namespace Internal {
/******************************************************************************
 *  Condition
 ******************************************************************************/
Condition::Condition(Handle hdl, const ParameterizedLocalizableText& text) :
    m_hdl(hdl),
    m_text(text)
{
}

/******************************************************************************
 *  IsTrue
 ******************************************************************************/
bool Condition::IsTrue() const
{
    Operand m_leftOperand(GetLeftOperandHandle(), m_text);
    Operand m_rightOperand(GetRightOperandHandle(), m_text);
    bool isTrue;

    switch (GetOperator().GetType()) {
        case Operator::Less:
            isTrue = (m_leftOperand < m_rightOperand);
            break;

        case Operator::LessOrEqual:
            isTrue = (m_leftOperand <= m_rightOperand);
            break;

        case Operator::Equal:
            isTrue = (m_leftOperand == m_rightOperand);
            break;

        case Operator::NotEqual:
            isTrue = (m_leftOperand != m_rightOperand);
            break;

        default:
            isTrue = false;
            break;
    }

    return isTrue;
}

/******************************************************************************
 *  Operand
 ******************************************************************************/
Operand::Operand(Handle hdl, const ParameterizedLocalizableText& text) : m_hdl(hdl), m_text(text) {
    if (GetOperandType() == Constant) {
        const OperandValue& value = GetData().m_value;

        switch (GetValueType()) {
        case Int32ValueType:
            m_value = DataItemValue(DataItemType<Int32>::Ident(), const_cast<Int32*>(value.GetValue<Int32>()), false);
            break;

        case FloatValueType:
            m_value = DataItemValue(DataItemType<Float>::Ident(), const_cast<Float*>(value.GetValue<Float>()), false);
            break;

        case StringValueType:
            m_strValue = String(value.GetValue<const TChar>());
            m_value = DataItemValue(DataItemType<String>::Ident(), &m_strValue, false);
            break;

        case BooleanValueType:
            m_value = DataItemValue(DataItemType<bool>::Ident(), const_cast<bool*>(value.GetValue<bool>()), false);
            break;

        default:
            break;
        }
    }
}

 /******************************************************************************
  *  operator<
  ******************************************************************************/
bool Operand::operator<(const Operand& rhs) const
{
    FEATSTD_DEBUG_ASSERT(GetValueType() == rhs.GetValueType());

    const DataItemValue& leftValue = GetValue();
    const DataItemValue& rightValue = rhs.GetValue();
    bool isLess;

    switch (GetValueType()) {
        case Int32ValueType:
            isLess = *leftValue.GetValue<Int32>() < *rightValue.GetValue<Int32>();
            break;

        case FloatValueType:
            isLess = *leftValue.GetValue<Float>() < *rightValue.GetValue<Float>();
            break;

        case StringValueType:
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker leftLock(leftValue.GetValue<String>()->GetCriticalSection());
                FeatStd::Internal::CriticalSectionLocker rightLock(rightValue.GetValue<String>()->GetCriticalSection());
#endif
                isLess = TextEncoding::Compare(leftValue.GetValue<String>()->GetCString(), rightValue.GetValue<String>()->GetCString()) < 0;
            }
            break;

        case BooleanValueType:
            isLess = *leftValue.GetValue<bool>() < *rightValue.GetValue<bool>();
            break;

        default:
            isLess = false;
            break;
    }

    return isLess;
}

/******************************************************************************
 *  operator<=
 ******************************************************************************/
bool Operand::operator<=(const Operand& rhs) const
{
    FEATSTD_DEBUG_ASSERT(GetValueType() == rhs.GetValueType());

    const DataItemValue& leftValue = GetValue();
    const DataItemValue& rightValue = rhs.GetValue();
    bool isLessOrEqual;

    switch (GetValueType()) {
        case Int32ValueType:
            isLessOrEqual = *leftValue.GetValue<Int32>() <= *rightValue.GetValue<Int32>();
            break;

        case FloatValueType:
            isLessOrEqual = *leftValue.GetValue<Float>() <= *rightValue.GetValue<Float>();
            break;

        case StringValueType:
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker leftLock(leftValue.GetValue<String>()->GetCriticalSection());
                FeatStd::Internal::CriticalSectionLocker rightLock(rightValue.GetValue<String>()->GetCriticalSection());
#endif
                isLessOrEqual = TextEncoding::Compare(leftValue.GetValue<String>()->GetCString(), rightValue.GetValue<String>()->GetCString()) <= 0;
            }
            break;

        case BooleanValueType:
            isLessOrEqual = *leftValue.GetValue<bool>() <= *rightValue.GetValue<bool>();
            break;

        default:
            isLessOrEqual = false;
            break;
    }

    return isLessOrEqual;
}

/******************************************************************************
 *  operator==
 ******************************************************************************/
bool Operand::operator==(const Operand& rhs) const
{
    FEATSTD_DEBUG_ASSERT(GetValueType() == rhs.GetValueType());

    const DataItemValue& leftValue = GetValue();
    const DataItemValue& rightValue = rhs.GetValue();
    bool isEqual;

    switch (GetValueType()) {
        case Int32ValueType:
            isEqual = *leftValue.GetValue<Int32>() == *rightValue.GetValue<Int32>();
            break;

        case FloatValueType:
            //lint -e{777}      Testing floats for equality
            isEqual = *leftValue.GetValue<Float>() == *rightValue.GetValue<Float>();
            break;

        case StringValueType:
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker leftLock(leftValue.GetValue<String>()->GetCriticalSection());
                FeatStd::Internal::CriticalSectionLocker rightLock(rightValue.GetValue<String>()->GetCriticalSection());
#endif
                isEqual = TextEncoding::Compare(leftValue.GetValue<String>()->GetCString(), rightValue.GetValue<String>()->GetCString()) == 0;
            }
            break;

        case BooleanValueType:
            isEqual = *leftValue.GetValue<bool>() == *rightValue.GetValue<bool>();
            break;

        default:
            isEqual = false;
            break;
    }

    return isEqual;
}

/******************************************************************************
 *  operator!=
 ******************************************************************************/
bool Operand::operator!=(const Operand& rhs) const
{
    FEATSTD_DEBUG_ASSERT(GetValueType() == rhs.GetValueType());

    const DataItemValue& leftValue = GetValue();
    const DataItemValue& rightValue = rhs.GetValue();
    bool isNotEqual;

    switch (GetValueType()) {
        case Int32ValueType:
            isNotEqual = *leftValue.GetValue<Int32>() != *rightValue.GetValue<Int32>();
            break;

        case FloatValueType:
            //lint -e{777}      Testing floats for equality
            isNotEqual = *leftValue.GetValue<Float>() != *rightValue.GetValue<Float>();
            break;

        case StringValueType:
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker leftLock(leftValue.GetValue<String>()->GetCriticalSection());
                FeatStd::Internal::CriticalSectionLocker rightLock(rightValue.GetValue<String>()->GetCriticalSection());
#endif
                isNotEqual = TextEncoding::Compare(leftValue.GetValue<String>()->GetCString(), rightValue.GetValue<String>()->GetCString()) != 0;
            }
            break;

        case BooleanValueType:
            isNotEqual = *leftValue.GetValue<bool>() != *rightValue.GetValue<bool>();
            break;

        default:
            isNotEqual = false;
            break;
    }

    return isNotEqual;
}

/******************************************************************************
 *  GetValue
 ******************************************************************************/
const Courier::DataItemValue& Operand::GetValue() const
{
    if (GetOperandType() == Constant) {
        return m_value;
    }
    else {
        const OperandValue& value = GetData().m_value;
        UInt8 paramIndex = static_cast<UInt8>(*value.GetValue<Int32>());
        return m_text.GetParameterValue(paramIndex);
    }
}

/******************************************************************************
 *  ResolveConditions
 ******************************************************************************/
void ConditionalFormatPattern::ResolveConditions() const
{
    const UInt8 conditionCount = GetConditionCount();
    Handle formatPatternHdl = GetDefaultFormatPatternHandle();

    // Check all condition. The first condition which is true defines the format pattern to be used.
    for (UInt8 i = 0; i < conditionCount; i++) {
        Condition condition(GetConditionHandle(i), m_text);

        if (condition.IsTrue()) {
            formatPatternHdl = condition.GetFormatPatternHandle();
            break;
        }
    }

    FormatPattern formatPattern(formatPatternHdl, m_text);
    formatPattern.FormatString();
}

/******************************************************************************
 *  FormatString
 ******************************************************************************/
void FormatPattern::FormatString()
{
    const TChar* formatStr = GetString();
    TChar* buffer = m_text.m_buffer;
    SizeType formatStrPos = 0;
    UInt32 lastParamPos = 0;
    UInt32 codePointsLeft = TextEncoding::CodePointCount(formatStr);

    const UInt8 paramCount = GetParameterCount();
    for (UInt8 i = 0; i < paramCount; i++) {
        UInt16 paramPos = GetParameterPosition(i); // this is the code-point position, NOT the byte position

        UInt32 codePointsToCopy = paramPos - lastParamPos;
        codePointsLeft -= codePointsToCopy;

        SizeType spaceLeft = m_text.GetMaxStringSize() - (buffer - m_text.m_buffer + sizeof(TChar));
        SizeType bytesCopied = TextEncoding::Copy(buffer, spaceLeft, &formatStr[formatStrPos], codePointsToCopy);

        buffer += bytesCopied;
        formatStrPos += bytesCopied;

        // if the param index is valid, insert the formatted parameter value
        UInt8 paramIndex = GetParameterIndex(i);
        UInt16 len = m_text.Format(paramIndex, buffer);
        buffer += len;

        lastParamPos = paramPos;
    }

    // and the rest of the string
    if (codePointsLeft > 0) {
        SizeType spaceLeft = static_cast<SizeType>(m_text.GetMaxStringSize()) - (buffer - m_text.m_buffer + sizeof(TChar));
        TextEncoding::Copy(buffer, spaceLeft, &formatStr[formatStrPos], codePointsLeft);
    }
}
}}}   // namespace Candera::Globalization::Internal
