//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_TRIGGER_BEHAVIOR_H
#define CANDERA_TRIGGER_BEHAVIOR_H

#include <CanderaBehavior/BehaviorBase/ConditionBehavior.h>
#include <CanderaBehavior/BehaviorBase/ActionBehavior.h>

namespace Candera {
/** @addtogroup BehaviorBase
  * @{
  */

  /**
 * @brief The TriggerBehavior references a ConditionBehavior and an array of ActionBehavior. 
 * ConditionBehavior and ActionBehavior are base classes. Any input event is processed by a TriggerBehavior, 
 * which evaluates a condition (ConditionBehavior). 
 * If the condition evaluates to true, then the TriggerBehavior invokes one or more ActionBehavior. 
 * A filter on event type can be achieved by a specialized ConditionBehaviorclass for such an event.
 **/
class TriggerBehavior : public ::Candera::Behavior {
public:
    typedef Candera::ConditionBehavior* ConditionType;

#if defined(CANDERA_LEGACY_BEHAVIORS_ENABLED)
    typedef Candera::ActionBehavior* ActionType;
#endif
    typedef ActionBehaviorArrayProperty ActionsType;
    typedef bool ConsumeEventType;

    /**
     * TriggerBehavior default constructor.
     */
    TriggerBehavior() :
        m_Condition(ConditionType()),
#if defined(CANDERA_LEGACY_BEHAVIORS_ENABLED)
        m_Action(ActionType()),
#else
        m_Actions(ActionsType()),
#endif
        m_consumeEvent(true)
    {
    }

    /**
     * TriggerBehavior destructor.
     */
    virtual ~TriggerBehavior()
    {
    }

    /**
     * Set the TriggerBehavior's condition.
     * @param condition The condition to set.
     */
    void SetCondition(const ConditionType& condition);

    /**
     * const method to retrieve the TriggerBehavior's const condition\&.
     * @return The TriggerBehavior's condition.
     */
    const ConditionType& GetCondition() const
    {
        return m_Condition;
    }

    /**
     * Retrieve the TriggerBehavior's condition.
     * @return The TriggerBehavior's condition.
     */
    ConditionType& GetCondition() {
        return m_Condition;
    }

#if defined(CANDERA_LEGACY_BEHAVIORS_ENABLED)
    /**
     * Set the TriggerBehavior's actions.
     * @param actions The actions to set.
     */
    void SetAction(const ActionType& actions);

    /**
     * const method to retrieve the TriggerBehavior's const action\&.
     * @return The TriggerBehavior's action.
     */
    const ActionType& GetAction() const
    {
        return m_Action;
    }

    /**
     * Retrieve the TriggerBehavior's action\&.
     * @return The TriggerBehavior's action.
     */
    ActionType& GetAction()
    {
        return m_Action;
    }
#else
    /**
     * Set the TriggerBehavior's actions.
     * @param actions The actions to set.
     */
    void SetActions(const ActionsType& actions);

    /**
     * const method to retrieve the TriggerBehavior's const action\&.
     * @return The TriggerBehavior's action.
     */
    const ActionsType& GetActions() const
    {
        return m_Actions;
    }
#endif
    /**
     * Retrieve the TriggerBehavior's action\&.
     * @return The TriggerBehavior's action.
     */
    ActionsType& GetActions()
    {
        return m_Actions;
    }

    /**
     * Enable or disable the consumption of events. Consumed events are not dispatched any further.
     * @param consumeEvent True if events should be consumed, false if events should be dispatched further.
     */
    void SetConsumeEvent(ConsumeEventType consumeEvent);

    /**
     * Check if the behavior is set to consume events. Consumed events are not dispatched any further.
     * @return True if events are set to be consumed, false if the events are set to be dispatched further.
     */
    ConsumeEventType GetConsumeEvent() const
    {
        return m_consumeEvent;
    }

protected:
    /**
     * Event handling method implementation with mechanisms to route events to other behaviors.
     * @param event The \event to handle.
     * @param dispatchResult The result.
     */
    virtual void OnEvent(const FeatStd::Event& event, EventDispatchResult& dispatchResult);

private:
    /// @cond Doxygen ignore - start
    // @WidgetBaseClass
    CdaBehaviorMixedDef(TriggerBehavior, ::Candera::Behavior)
        CdaDescription("Passes a value to a Condition Behavior and if the response is positive, invokes an Action behavior.")
        CdaReadableName("Event Handler")
        CdaCategory("Control")
        CdaProperties()
            CdaProperty(Condition, ConditionType, GetCondition, SetCondition)
                CdaDescription("The ConditionBehavior that TriggerBehavior passes a value to and expect a response from to see if Action needs to be invoked.")
            CdaPropertyEnd()
#if defined(CANDERA_LEGACY_BEHAVIORS_ENABLED) 
            CdaProperty(Action, ActionType, GetAction, SetAction)
            CdaDescription("The ActionBehavior that TriggerBehavior invokes in the even that it receives a positive response from Condition.")
            CdaPropertyEnd()
#else
            CdaProperty(Actions, ActionsType, GetActions, SetActions)
                CdaDescription("The set of ActionBehaviors that TriggerBehavior invokes in the even that it receives a positive response from Condition.")
            CdaPropertyEnd()
#endif
            CdaProperty(ConsumeEvent, ConsumeEventType, GetConsumeEvent, SetConsumeEvent)
                CdaDescription("Set this flag to false if the event should not be consumed by the trigger. By default it is set to true and if the trigger evaluates to true the event will not be routed to any other behavior.")
            CdaPropertyEnd()
        CdaPropertiesEnd()
    CdaBehaviorDefEnd()
    /// @endcond Doxygen ignore - end

    ConditionType m_Condition;
#if defined(CANDERA_LEGACY_BEHAVIORS_ENABLED)
    ActionType m_Action;
#endif
    ActionsType m_Actions;

    bool m_consumeEvent;
};
/** @} */
} // namespace Candera

#endif
