//########################################################################
// (C)  Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################
#ifndef CANDERA_PROCESS_VALUE_BEHAVIOR_H
#define CANDERA_PROCESS_VALUE_BEHAVIOR_H

#include <CanderaBehavior/BehaviorBase/Behavior.h>
#include <CanderaAssetLoader/AssetLoaderBase/ArrayProperty.h>
#include <CanderaAssetLoader/AssetLoaderBase/ArrayDataType.h>
#include <FeatStd/Util/Variant.h>
#include <FeatStd/Util/Optional.h>

namespace Candera {
/** @addtogroup BehaviorBase
  * @{
  */
    class ProcessValueBehavior;
    /**
    * @brief Event gets sent by Source Process Value Behaviors (Value Behavior) when value changed.
    * \event ProcessValueEvent - Value Processed
    **/
    class ProcessValueEvent : public FeatStd::Event
    {
    public:
        FEATSTD_RTTI_DECLARATION();

        /**
         * ProcessValueEvent constructor.
         * @param variant
         * @param sender The event's sender.
         * @param minimum The minimum value.
         * @param maximum The maximum value.
         */
        ProcessValueEvent(const FeatStd::Variant& variant, const ProcessValueBehavior& sender,
            FeatStd::Optional< FeatStd::Variant > minimum, FeatStd::Optional< FeatStd::Variant > maximum) :
            m_variant(variant),
            m_sender(sender),
            m_minimum(minimum),
            m_maximum(maximum)
        {
        }

#if defined(CANDERA_LEGACY_BEHAVIORS_ENABLED)
        /**
         * ProcessValueEvent constructor.
         * @param variant
         * @param sender The event's sender.
         */
        ProcessValueEvent(const FeatStd::Variant& variant, const ProcessValueBehavior& sender) :
            m_variant(variant),
            m_sender(sender),
            m_minimum(),
            m_maximum()
        {
        }
#endif

        /**
         * ProcessValueEvent destructor.
         */
        virtual ~ProcessValueEvent()
        {
        }

        /**
         * Retrieves the variant.
         * @return m_variant FeatStd::Variant
         */
        const FeatStd::Variant& GetVariant() const
        {
            return m_variant;
        }

        /**
         * Retrieves the sender.
         * @return m_sender ProcessValueBehavior&
         */
        const ProcessValueBehavior& GetSender() const
        {
            return m_sender;
        }
        /**
         * Retrieves the minimum value.
         * @return m_minimum FeatStd::Optional<FeatStd::Variant>
         */
        FeatStd::Optional<FeatStd::Variant> GetMinimum() const
        {
            return m_minimum;
        }
        /**
         * retrieves the maximum value.
         * @return m_maximum FeatStd::Optional<FeatStd::Variant>
         */
        FeatStd::Optional<FeatStd::Variant> GetMaximum() const
        {
            return m_maximum;
        }

    protected:
        /**
         * Check if this Behavior is a direct event receiver.
         * @return True if this Behavior is a direct event receiver.
         */
        virtual bool IsDirectEventReceiver() const
        {
            return false;
        }

    private:
        FEATSTD_MAKE_CLASS_STATIC(ProcessValueEvent);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(ProcessValueEvent);

        FeatStd::Variant m_variant;
        const ProcessValueBehavior& m_sender;
        FeatStd::Optional<FeatStd::Variant> m_minimum;
        FeatStd::Optional<FeatStd::Variant> m_maximum;
    };

#if defined(CANDERA_LEGACY_BEHAVIORS_ENABLED)
    class ProcessValueResult : public EventDispatchResult
    {
    public:
        FEATSTD_RTTI_DECLARATION();

        /**
         * ProcessValueEvent default constructor.
         */
        ProcessValueResult() { m_variant = FeatStd::Variant(Float(0)); }

        /**
         * ProcessValueEvent constructor.
         * @param variant
         */
        ProcessValueResult(const FeatStd::Variant& variant) : m_variant(variant) {}

        /**
         * ProcessValueEvent destructor.
         */
        virtual ~ProcessValueResult() {}

        /**
         * Sets the variant.
         * @param variant
         */
        void SetVariant(const FeatStd::Variant& variant) {
            m_variant = FeatStd::Variant(variant);
        }

        /**
         * Retrieves the variant.
         * @return m_variant FeatStd::Variant
         */
        FeatStd::Variant& GetVariant() {
            return m_variant;
        }

    private:
        FeatStd::Variant m_variant;
    };
#endif

    /**
     * @brief The ProcessValueBehavior provides a Variant data type which is bindable. 
     * A value change results in a processing of that value by sending a ProcessValueEvent to a 
     * referenced ProcessValueBehavior. 
     * ProcessValueBehavior is the base class. Derived classes can be chained to realize 
     * a data flow processing from input to an appropriate (visual) output. 
     * Example: from a changed "speed" value to a mapped rotation value and further to an 
     * animated rotation of a needle node.
     */
    class ProcessValueBehavior : public ::Candera::Behavior
    {
    public:
        /**
         * ProcessValueBehavior default constructor.
         */
        ProcessValueBehavior()
        {
        }

        /**
         * ProcessValueBehavior destructor.
         */
        virtual ~ProcessValueBehavior()
        {
        }

    private:
        /// @cond Doxygen ignore - start
        // @WidgetBaseClass
        CdaBehaviorMixedDef(ProcessValueBehavior, ::Candera::Behavior)
            CdaDescription("")
            CdaReadableName("Process Value")
            CdaProperties()
            CdaPropertiesEnd()
        CdaBehaviorDefEnd()
        /// @endcond Doxygen ignore - end
    };

    typedef ArrayProperty<ProcessValueBehavior*> ProcessValueBehaviorArrayProperty;
/** @} */
} // namespace Candera

CGI_BEHAVIOR_FORWARD_AssetLoaderDataTypeDef(::Candera::ProcessValueBehavior)

#endif
