//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SceneContextBase.h"

#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <CanderaWidget/WidgetBase/WidgetBase.h>

#include <CanderaPlatform/OS/StringPlatform.h>
#include <CanderaAssetLoader/AssetLoaderBase/ContentLoader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {
    using namespace Candera::Internal;
    using namespace MemoryManagement;

    const Char* SceneContextBase::dummyIdentifier = 0;

    SceneContextBase::SceneContextBase() :
        m_loadedThemeId(),
        m_defaultThemeId(),
        m_uploadPolicy(ContentLoader::NoVramUploadPolicy),
        m_currentEntry(m_reloadList.Begin())
    {
    }


    SceneContextBase::SceneContextBase(const SceneContextBase& other) :
        m_loadedThemeId(other.m_loadedThemeId),
        m_defaultThemeId(other.m_defaultThemeId),
        m_uploadPolicy(other.m_uploadPolicy),
        m_currentEntry(m_reloadList.Begin())
    {
    }

    SceneContextBase::~SceneContextBase(void)
    {
    }

    bool SceneContextBase::GetFirstWidget(WidgetBase*& widget, const Char*& identifier) const
    {
        m_widgetListIterator = GetWidgetIterator();
        if (m_widgetListIterator.IsValid()) {
            widget = *m_widgetListIterator;
            identifier = widget->GetStringId()->GetId();
            return true;
        }

        return false;
    }

    bool SceneContextBase::GetNextWidget(WidgetBase*& widget, const Char*& identifier) const
    {
        if (m_widgetListIterator.IsValid()) {
            ++m_widgetListIterator;
            if (m_widgetListIterator.IsValid()) {
                widget = *m_widgetListIterator;
                identifier = widget->GetStringId()->GetId();
                return true;
            }
        }

        return false;
    }

    WidgetBase* SceneContextBase::GetWidgetByIdentifer(const Char* identifier) const
    {
        if (identifier == 0) {
            return 0;
        }

        for (WidgetIterator iterator = GetWidgetIterator(); iterator.IsValid(); ++iterator) {
            if (StringPlatform::CompareStrings(identifier, (*iterator)->GetStringId()->GetId()) == 0) {
                return *iterator;
            }
        }

        return 0;
    }

    WidgetBase* SceneContextBase::GetWidget(Id id) const
    {
        for (WidgetIterator iterator = GetWidgetIterator(); iterator.IsValid(); ++iterator) {
            if ((*iterator)->GetId() == id) {
                return *iterator;
            }
        }

        return 0;
    }

    bool SceneContextBase::AddWidget(WidgetBase* widget)
    {
        if ((widget == 0) || (widget->GetName() == 0)) {
            return false;
        }

        return m_widgetList.Add(widget);
    }

    SceneContextBase::WidgetIterator SceneContextBase::GetWidgetIterator() const
    {
        return WidgetIterator(m_widgetList.ConstBegin(), m_widgetList.ConstEnd());
    }


    void SceneContextBase::AddAnimatedPropertyAttachment(Id hostId, MemoryManagement::SharedPointer<Animation::AnimationBlendedProperty> attachment)
    {
        for (Vector<AnimationBlendedPropertyList>::Iterator it = m_animationBlendedPropertyLists.Begin(); it != m_animationBlendedPropertyLists.End(); ++it) {
            if ((*it).m_animationId == hostId) {
                static_cast<void>((*it).m_properties.Add(attachment));
                return;
            }
        }
        AnimationBlendedPropertyList newAnimation;
        newAnimation.m_animationId = hostId;
        static_cast<void>(newAnimation.m_properties.Add(attachment));
        static_cast<void>(m_animationBlendedPropertyLists.Add(newAnimation));
    }

    const Char* SceneContextBase::GetAttachedAnimationName(UInt32 index)
    {
        return DefaultAssetProvider::GetInstance().GetNameById(AnimationLib, GetAttachedAnimationId(index), 0);
    }

    UInt32 SceneContextBase::GetAnimatedPropertyAttachmentCount(Id animationId)
    {
        for (Vector<AnimationBlendedPropertyList>::Iterator it = m_animationBlendedPropertyLists.Begin(); it != m_animationBlendedPropertyLists.End(); ++it) {
            if ((*it).m_animationId == animationId) {
                return FeatStd::Internal::NumericConversion<UInt32>((*it).m_properties.Size());
            }
        }

        return 0;
    }


    Animation::AnimationBlendedProperty::SharedPointer SceneContextBase::GetAnimatedPropertyAttachment(Id assetId, UInt32 index)
    {
        for (Vector<AnimationBlendedPropertyList>::Iterator it = m_animationBlendedPropertyLists.Begin(); it != m_animationBlendedPropertyLists.End(); ++it) {
            if ((*it).m_animationId == assetId) {
                if (index < FeatStd::Internal::NumericConversion<UInt32>((*it).m_properties.Size())) {
                    return (*it).m_properties[index];
                }
            }
        }

        return Animation::AnimationBlendedProperty::SharedPointer();
    }

    Id SceneContextBase::GetAttachedAnimationId(UInt32 index)
    {
        if (static_cast<SizeType>(index) < m_animationBlendedPropertyLists.Size()) {
            return m_animationBlendedPropertyLists[index].m_animationId;
        }

        return 0;
    }

    const Char* SceneContextBase::GetLoadedThemeName() const
    {
        return DefaultAssetProvider::GetInstance().GetNameById(ThemeLib, m_loadedThemeId, 0);
    }

    const Char* SceneContextBase::GetDefaultThemeName() const
    {
        return DefaultAssetProvider::GetInstance().GetNameById(ThemeLib, m_defaultThemeId, 0);
    }

}

