//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RELOADENTRY_H)
#define CANDERA_RELOADENTRY_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderMemoryPool.h>
#include <Candera/System/Container/SingleLinkedList.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetId.h>
#include <CanderaPlatform/OS/StringPlatform.h>


namespace Candera {
    namespace MetaInfo {
        template <typename T>
        class PropertyMetaInfo;
    }
    class DefaultAssetProvider;
    namespace Internal {

        class DependencyLoaderBase;
        typedef Candera::MemoryManagement::SharedPointer<DependencyLoaderBase> DependencyEntry;
        typedef SingleLinkedList<DependencyEntry, ASSETLOADER_TRANSIENT_ALLOCATOR> DependencyList;

        class DependencyLoaderBase {
            public:
                DependencyLoaderBase() {};
                virtual ~DependencyLoaderBase() {}
                virtual bool Load(DependencyList* dependencyList) = 0;

            private:
                FEATSTD_MAKE_CLASS_UNCOPYABLE(DependencyLoaderBase);
                CANDERA_SHARED_POINTER_DECLARATION();
        };

        template <typename T>
        class TypedDependencyLoader : public DependencyLoaderBase {
            public:
                typedef bool (*ReloadFn)(T&, AssetId id, DependencyList*, const DefaultAssetProvider*);

                TypedDependencyLoader(T& object, AssetId id, ReloadFn reloadFn, DefaultAssetProvider* provider);

                virtual ~TypedDependencyLoader();

                virtual bool Load(DependencyList* dependencyList);

            private:
                T& m_object;
                ReloadFn m_reloadFn;
                AssetId m_id;
                DefaultAssetProvider* m_provider;
        };

        template<typename T>
        TypedDependencyLoader<T>::TypedDependencyLoader(T& object, AssetId id, ReloadFn reloadFn, DefaultAssetProvider* provider) :
            m_object(object),
            m_reloadFn(reloadFn),
            m_id(id),
            m_provider(provider)
        {
        }

        template<typename T>
        TypedDependencyLoader<T>::~TypedDependencyLoader()
        {
        }

        template<typename T>
        bool TypedDependencyLoader<T>::Load(DependencyList* dependencyList)
        {
            return m_reloadFn(m_object, m_id, dependencyList, m_provider);
        }

        template <typename T, typename PMI>
        class MetaInfoPropertyDependencyLoader : public DependencyLoaderBase {
            public:
                typedef bool (*ReloadFn)(T&, PMI&, const Char*);

                MetaInfoPropertyDependencyLoader(T& object, PMI& propertyMetaInfo, const Char* value, ReloadFn reloadFn);

                virtual ~MetaInfoPropertyDependencyLoader();

                virtual bool Load(DependencyList* dependencyList);

            private:
                Char* m_value;
                T& m_object;
                PMI& m_propertyMetaInfo;
                ReloadFn m_reloadFn;
        };

        template <typename T, typename PMI>
        MetaInfoPropertyDependencyLoader<T, PMI>::MetaInfoPropertyDependencyLoader(T& object, PMI& propertyMetaInfo, const Char* value, ReloadFn reloadFn) :
            m_value(0),
            m_object(object),
            m_propertyMetaInfo(propertyMetaInfo),
            m_reloadFn(reloadFn)
        {
            if (value != 0) {
                m_value = ASSETLOADER_TRANSIENT_NEW_ARRAY(Char, StringPlatform::Length(m_value) + 1);
                StringPlatform::Copy(m_value, value);
            }
        }

        template <typename T, typename PMI>
        MetaInfoPropertyDependencyLoader<T, PMI>::~MetaInfoPropertyDependencyLoader()
        {
            if (m_value != 0) {
                ASSETLOADER_DELETE_ARRAY(m_value);
                m_value = 0;
            }
        }

        template <typename T, typename PMI>
        bool MetaInfoPropertyDependencyLoader<T, PMI>::Load(DependencyList* /*dependencyList*/)
        {
            return m_reloadFn(m_object, m_propertyMetaInfo, m_value);
        }

    }
}

#endif
