//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_OBJECTMAP_H)
#define CANDERA_OBJECTMAP_H

#include <Candera/Environment.h>
#include <Candera/System/Container/Vector.h>
#include <CanderaPlatform/OS/StringPlatform.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderMemoryPool.h>

namespace Candera {
        namespace Internal {
        /**
         * @brief Encapsulates a key-value pair for ObjectMap
         * The key is of type string (const Char*)
         */
        template <typename T>
        class MapPair
        {
        public:
            MapPair(const Char* key, bool duplicateKey, T value):
                m_value(value),
                m_key(key),
                m_isKeyDuplicated(duplicateKey)
            {
                if (duplicateKey) {
                    m_key = DuplicateKey(m_key);
                }
            };

            MapPair(const Char* key, bool duplicateKey) :
                m_key(key),
                m_isKeyDuplicated(duplicateKey)
            {
                m_value = T();
                if (duplicateKey) {
                    m_key = DuplicateKey(m_key);
                }
            };

            MapPair(const MapPair& other) {
                this->m_isKeyDuplicated = other.m_isKeyDuplicated;
                this->m_value = other.m_value;
                if (this->m_isKeyDuplicated) {
                    m_key = DuplicateKey(other.m_key);
                }
                else
                    this->m_key = other.m_key;
            }

            ~MapPair() {
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1551, Candera::MapPair::~MapPair, none of these operations throw)
                //lint -e{1551} exceptions disabled in whole Candera build
                if (m_isKeyDuplicated) {
                    ASSETLOADER_DELETE_ARRAY(m_key);
                }
                m_key = 0;
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1540, Candera::MapPair<T>::m_value, CANDERA_LINT_REASON_ASSOCIATION)
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1740, Candera::MapPair<T>::m_value, CANDERA_LINT_REASON_ASSOCIATION)
            } //lint !e1740 members under external control or static

            MapPair& operator=(const MapPair& other) {
                if (this->m_isKeyDuplicated && m_key != 0) {
                    ASSETLOADER_DELETE_ARRAY(m_key);
                    m_key = 0;
                }
                this->m_isKeyDuplicated = other.m_isKeyDuplicated;
                this->m_value = other.m_value;
                if (this->m_isKeyDuplicated) {
                    m_key = DuplicateKey(other.m_key);
                }
                else
                    this->m_key = other.m_key;
                return *this;
            }

            const Char* GetKey() const { return m_key; }
            T& GetValue() { return m_value; } // To discus: is it ok ? Allows assignment
            const T& GetValue() const { return m_value; }

        protected:
            const Char* DuplicateKey(const Char* key) const
            {
                UInt32 keyLength = 0;
                while (key[keyLength] != '\0') keyLength++;
                keyLength++; // Include \0
                const Char* result = ASSETLOADER_TRANSIENT_NEW_ARRAY(Char, keyLength);
                if (result != 0) {
                    MemoryPlatform::Copy((void*)const_cast<Char*>(result), (void*)const_cast<Char*>(key), keyLength);
                }
                return result;
            }
        private:
            T m_value;
            const Char* m_key;
            bool m_isKeyDuplicated;
        };

        /**
         * @brief Encapsulates a "STL Map"-like table excepting that the keys are always of
         * type string ( const Char* )
         *
         * The constructor accepts a bool "duplicateKeys" to indicate whether or not the keys
         * of type "const Char*" should be kept as copies (the string value) during key-value
         * pair lifetime in the object map. By default, the object map doesn't duplicate
         * the keys so the key lifetime should be ensured and managed by application otherwise.
         *
         * <b>Updates: </b><br>
         * - <b> v1.2: </b>
         *     - Added method:
         *       @code IsIteratorValid(const Iterator& i) @endcode
         *       To be used to validate iterators
         *       in constructions such as @code if (i != map.End()) @endcode Replacement: @code if (map.IsIteratorValid(i)) @endcode
         *       The following construction still works:
         *       @code if (i != map.End()) @endcode
         *        <br>
         *       Usage code-sample:
         *       @code
         *       ObjectMap<MemoryManagement::SharedPointer<Shader> > map;
         *       ObjectMap<MemoryManagement::SharedPointer<Shader> >::Iterator i;
         *
         *       MemoryManagement::SharedPointer<Shader> sh1 = Shader::Create();
         *       MemoryManagement::SharedPointer<Shader> sh2 = Shader::Create();
         *       MemoryManagement::SharedPointer<Shader> sh3 = Shader::Create();
         *
         *       map["one"] = sh1;
         *       map["two"] = sh2;
         *
         *       MemoryManagement::SharedPointer<Shader> sh1Copy = map["one"];
         *
         *       i = map.Find("one");
         *       FEATSTD_DEBUG_ASSERT(map.IsIteratorValid(i));  // Found
         *
         *       i = map.Find("three");
         *       FEATSTD_DEBUG_ASSERT(!map.IsIteratorValid(i)); // Not found
         *
         *       MemoryManagement::SharedPointer<Shader> null(0);
         *
         *       for (i = map.Begin(); map.IsIteratorValid(i); i++) {
         *          MemoryManagement::SharedPointer<Shader> sh = i->GetValue();
         *          const Char* name = i->GetKey();
         *
         *          i->GetValue() = null;                // Allows assignment.
         *       }
         *
         *       ObjectMap<MemoryManagement::SharedPointer<Shader> >::ConstIterator ci = map.ConstBegin();
         *       for (ci = map.ConstBegin(); map.IsIteratorValid(ci); ci++) {
         *          MemoryManagement::SharedPointer<Shader> sh = ci->GetValue();
         *          const Char* name = ci->GetKey();
         *          const MemoryManagement::SharedPointer<Shader> shc = ci->GetValue();
         *       }
         *       @endcode
         */
        template <typename T>
        class ObjectMap
        {
        public:

            class MapIterator {
            public:
                MapIterator() : m_nameMap(0), m_index(-1) { };

                bool operator==(const MapIterator& i2) const
                {
                    return i2.m_index == m_index;
                }

                bool operator!=(const MapIterator& i2) const
                {
                    return i2.m_index != m_index;
                }

                MapPair<T>& operator*() const
                {
                    FEATSTD_DEBUG_ASSERT(m_nameMap != 0);
                    return (*m_nameMap)[m_index];
                }

                MapPair<T>* operator->() const
                {
                    return &(*m_nameMap)[m_index];
                }

                MapIterator operator++(Int /*ofst*/)
                {
                    if (m_nameMap == 0) {
                        m_index = 0;
                        return MapIterator(0, 0);
                    }
                    m_index++;
                    if (m_index > static_cast<Int>(m_nameMap->Size())) {
                        m_index = static_cast<Int>(m_nameMap->Size());
                    }
                    return MapIterator(m_nameMap, m_index);
                }

                MapIterator& operator++()
                {
                    if (m_nameMap == 0) {
                        m_index = 0;
                        return *this;
                    }
                    m_index++;
                    if (m_index > static_cast<Int>(m_nameMap->Size())) {
                        m_index = static_cast<Int>(m_nameMap->Size());
                    }
                    return *this;
                }

                MapIterator operator--(Int /*ofst*/)
                {
                    if (m_nameMap == 0) {
                        m_index = 0;
                        return MapIterator(0, 0);
                    }
                    m_index--;
                    if (m_index < 0) {
                        m_index = 0;
                    }
                    return MapIterator(m_nameMap, m_index);
                }

                MapIterator& operator--()
                {
                    if (m_nameMap == 0) {
                        m_index = 0;
                        return *this;
                    }
                    m_index--;
                    if (m_index < 0) {
                        m_index = 0;
                    }
                    return *this;
                }

            protected:
                MapIterator(ObjectMap* nameMap, Int index) : m_nameMap(nameMap), m_index(index) { }
                MapIterator(ObjectMap* nameMap) : m_nameMap(nameMap), m_index(-1)
                {
                    if (nameMap != 0) {
                        if (nameMap->Size() > 0) {
                            m_index = 0;
                        }
                    }
                }

            private:
                friend class ObjectMap;

                ObjectMap* m_nameMap;
                Int m_index;
            };

            class MapConstIterator {
            public:
                MapConstIterator(): m_nameMap(0), m_index(-1)  { };

                bool operator==(const MapConstIterator& i2) const
                {
                    return i2.m_index == m_index;
                }

                bool operator!=(const MapConstIterator& i2) const
                {
                    return i2.m_index != m_index;
                }

                const MapPair<T>& operator*() const
                {
                    FEATSTD_DEBUG_ASSERT(m_nameMap != 0);
                    return (*m_nameMap)[m_index];
                }

                const MapPair<T>* operator->() const
                {
                    return &(*m_nameMap)[m_index];
                }

                MapConstIterator operator++(Int /*ofst*/)
                {
                    if (m_nameMap == 0) {
                        m_index = 0;
                        return MapConstIterator(0, 0);
                    }
                    m_index++;
                    if (m_index > static_cast<Int>(m_nameMap->Size())) {
                        m_index = static_cast<Int>(m_nameMap->Size());
                    }
                    return MapConstIterator(m_nameMap, m_index);
                }

                MapConstIterator& operator++()
                {
                    if (m_nameMap == 0) {
                        m_index = 0;
                        return *this;
                    }
                    m_index++;
                    if (m_index > static_cast<Int>(m_nameMap->Size())) {
                        m_index = static_cast<Int>(m_nameMap->Size());
                    }
                    return *this;
                }

                MapConstIterator operator--(Int /*ofst*/)
                {
                    if (m_nameMap == 0) {
                        m_index = 0;
                        return MapConstIterator(0, 0);
                    }
                    m_index--;
                    if (m_index < 0) {
                        m_index = 0;
                    }
                    return MapConstIterator(m_nameMap, m_index);
                }

                MapConstIterator& operator--()
                {
                    if (m_nameMap == 0) {
                        m_index = 0;
                        return *this;
                    }
                    m_index--;
                    if (m_index < 0) {
                        m_index = 0;
                    }
                    return *this;
                }
            protected:
                MapConstIterator(const ObjectMap* nameMap, Int index) : m_nameMap(nameMap), m_index(index) { }
                MapConstIterator(const ObjectMap* nameMap) : m_nameMap(nameMap), m_index(-1)
                {
                    if (nameMap != 0) {
                        if (nameMap->Size() > 0) {
                            m_index = 0;
                        }
                    }
                }

            private:
                friend class ObjectMap;

                const ObjectMap* m_nameMap;
                Int m_index;
            };

            typedef MapIterator Iterator;
            typedef MapConstIterator ConstIterator;
            typedef MapPair<T> ObjectMapPair;

            ObjectMap(void) :
                m_duplicateKeys(true)
            {
                m_pairs.Reserve(4);
            }

            ObjectMap(bool duplicateKeys) :
                m_duplicateKeys(duplicateKeys)
            {
                m_pairs.Reserve(4);
            }

            ~ObjectMap(void) { };

            FeatStd::SizeType Size() const
            {
                return m_pairs.Size();
            }

            Iterator Begin()
            {
                return Iterator(this, 0);
            }

            ConstIterator ConstBegin() const
            {
                return ConstIterator(this, 0);
            }

            Iterator End()
            {
                return Iterator(this, static_cast<Int>(m_pairs.Size()));
            }

            ConstIterator ConstEnd() const
            {
                return ConstIterator(this, static_cast<Int>(m_pairs.Size()));
            }

            Iterator Find(const Char* key)
            {
                FeatStd::SizeType idx = m_pairs.Size();
                for (FeatStd::SizeType i = 0; i < m_pairs.Size(); i++) {
                    if (StringPlatform::CompareStrings(key, m_pairs[i].GetKey()) == 0) {
                        idx = i;
                        break;
                    }
                }
                return Iterator(this, static_cast<Int>(idx));
            }

            ConstIterator Find(const Char* key) const
            {
                FeatStd::SizeType idx = m_pairs.Size();
                for (FeatStd::SizeType i = 0; i < m_pairs.Size(); i++) {
                    if (StringPlatform::CompareStrings(key, m_pairs[i].GetKey()) == 0) {
                        idx = i;
                        break;
                    }
                }
                return ConstIterator(this, static_cast<Int>(idx));
            }

            bool IsIteratorValid(const Iterator& i) const {
                bool isValid = (i.m_index >= 0 && i.m_nameMap == this && static_cast<SizeType>(i.m_index) < Size());
                return isValid;
            }

            bool IsIteratorValid(const ConstIterator& i) const {
                bool isValid = (i.m_index >= 0 && i.m_nameMap == this && static_cast<SizeType>(i.m_index) < Size());
                return isValid;
            }

            MapPair<T>& operator[](Int idx)
            {
                return m_pairs[idx];
            }

            const MapPair<T>& operator[](Int idx) const
            {
                return m_pairs[idx];
            }

            T& operator[](const Char* key)
            {
                FeatStd::SizeType idx = m_pairs.Size();
                for (FeatStd::SizeType i = 0; i < m_pairs.Size(); i++) {
                    if (StringPlatform::CompareStrings(key, m_pairs[i].GetKey()) == 0) {
                        idx = i;
                        break;
                    }
                }
                if (idx < m_pairs.Size()) {
                    return m_pairs[idx].GetValue();
                } else {
                    m_pairs.Add(ObjectMapPair(key, m_duplicateKeys));
                    return m_pairs[m_pairs.Size() - 1].GetValue();
                }
            }

            Iterator Remove(const Iterator& iterator) {
                m_pairs.Remove(iterator.m_index);
                return Iterator(iterator.m_nameMap, iterator.m_index);
            }

            void Clear() {
                m_pairs.Clear();
            }

        private:
            Vector<ObjectMapPair, FeatStd::Internal::LinearIncreasePolicy<10>, ASSETLOADER_PERMANENT_ALLOCATOR> m_pairs;
            bool m_duplicateKeys;
        };
    }
}

#endif // CANDERA_OBJECTMAP_H
