//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MEMORYASSETREPOSITORY_H)
#define CANDERA_MEMORYASSETREPOSITORY_H

#include <CanderaAssetLoader/AssetLoaderBase/AssetRepository.h>

namespace Candera {

/** @addtogroup AssetLoaderBase
 *  @{
 */
 
    /**
     * @brief Class providing methods to read asset from memory(e.g. from flash memory)
     */
    class MemoryAssetRepository : public AssetRepository
    {
    public:
    
        /**
         * Constructor
         * @param memoryStart The address of the memory the asset is located
         */
        MemoryAssetRepository(Handle memoryStart);

        /**
         * Destructor
         */
        virtual ~MemoryAssetRepository(void);

        /**
         * Initializes the repository so it's ready for read
         * @return true if successful
         */
        virtual bool Initialize();
        
        /**
         * Reads from the asset repository and copies output in buffer. Position indicator (AssetTell) is
         * advanced by the total amount of bytes read.
         * @param buffer      Pre allocated buffer to copy read data into.
         * @param elementSize Size of one element to read.
         * @param count       Count of elements to read.
         * @return amount of bytes read
         */
        virtual SizeType AssetRead(void* buffer, SizeType elementSize, SizeType count);

        /**
         * Seeks a position in the asset by offset bytes
         * @param offset Number of offset to look and set position indicator (AssetTell) to.
         * @param mark   Whether offset shall be added to the begin or the current position of the stream.
         * @return true if successful
         */
        virtual bool AssetSeek(OffsetType offset, AssetSeekMark mark);
        
        /**
         * Returns the current asset repository position.
         * @return The current asset repository position.
         */
        virtual OffsetType AssetTell();
        
        /**
         * Returns the available asset access modes.
         * @return The available asset access modes.
         */
        virtual AssetAccessMode GetAssetAccessMode() { return ReadOnlyAddressable; }
        
        /**
         * Retrieves a pointer to a block within the asset reposition.
         * The block starts from the current seek position. The position and status flags
         * are updated the same as in AssetRead. In case of success the retrieved block
         * has the size elementSize * count. The return value specifies the number of elements
         * available within the block or -1 on failure.
         * @param buffer Pointer to an object where to store the block start address.
         * @param elementSize Size of an element within the block.
         * @param count Number of requested elements.
         * @return Number of elements successfully reserved. -1 on failure.
         */
        virtual SizeType GetConstData(const void ** buffer, SizeType elementSize, SizeType count);

        /**
         * Finalizes the repository.
         * @return true if successful
         */
        virtual bool Finalize();

    private:
        SizeType m_memoryStart;
        SizeType m_filePos;
    };
 
 /** @} */ // end of AssetLoaderBase
 
}

#endif // CANDERA_MEMORYASSETREPOSITORY_H
