//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TypedCffAssetProvider_H)
    #define CANDERA_TypedCffAssetProvider_H

#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/Generic/TypedAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetId.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {
    namespace Internal {

        template <typename T, typename CachingStrategy, typename TBuilder = AssetBuilder<T> >
        class TypedCffAssetProvider : public TypedAssetProvider<T, CachingStrategy, TBuilder> {
            FEATSTD_LOG_SET_REALM(Candera::Diagnostics::LogRealm::CanderaAssetLoader);
            public:
                TypedCffAssetProvider(DefaultAssetProvider* provider):TypedAssetProvider<T, CachingStrategy, TBuilder>(provider) {}
                /**
                 * Retrieve an object by name.
                 *
                 * @param name Name of the object.
                 * @return Object instance.
                 */
                T Get(AssetId id, DependencyList* dependencyList = 0);

                /**
                 * Retrieve an object by name.
                 *
                 * @param#1 id - Asset's id
                 * @param#2 dependencyList - Default value is null.
                 * @return Object's address or null pointer.
                 */
                T GetFromCache(AssetId id, DependencyList* dependencyList = 0);
        };

        template <typename T, typename CachingStrategy, typename TBuilder>
        T TypedCffAssetProvider<T, CachingStrategy, TBuilder>::Get(AssetId id, DependencyList* dependencyList)
        {
            CffLoaderContext context = {id, this->m_provider, AssetDataHandle(), dependencyList, false, 0};

#ifdef FEATSTD_THREADSAFETY_ENABLED
            //block cache access during object construction
            FeatStd::Internal::CriticalSectionLocker cacheLock(&this->m_cacheCriticalSection);
            //block theme changing during object construction
            FeatStd::Internal::CriticalSectionLocker themeLock(&this->m_provider->m_themeChangeCriticalSection);
#endif

            Id libraryId = AssetIdFunctions::GetLibraryId(id);
            T result = this->m_cachingStrategy.Get(libraryId);
            if (result != 0 && !TBuilder::IsValid(result, context)) {
                this->m_cachingStrategy.Remove(libraryId);
                result = T(0);
            }

            if (result == T(0)) {
                if (this->m_dependencyStack.Contains(libraryId)) {
                    FEATSTD_LOG_INFO("Cyclic dependency for object " AssetIdLogStr, AssetIdLogArgs(id));
                }
                else {
                    if (!this->m_dependencyStack.Add(libraryId)) {
                        FEATSTD_LOG_INFO("Cyclic dependency for object " AssetIdLogStr, AssetIdLogArgs(id));
                    }
                    else {
                        if (this->m_provider->m_assetGroup != 0) {
                            context.handle = this->m_provider->m_assetGroup->GetItemDataHandle(id, &context.repositoryId);
                            if (!context.IsValid()) {
                                if (id.m_libraryType != 0) {
                                    //workaround for AnimationPlayer(0), it could be that actually an AnimationGroupPlayer(1) is searched. An error will be thrown then if not found.
                                    FEATSTD_LOG_INFO("Invalid asset handle for library object " AssetIdLogStr, AssetIdLogArgs(id));
                                }
                            }
                            else {
                                result = TBuilder::CreateAndBuildFirstPass(context);
                                if (result == T(0)) {
                                    FEATSTD_LOG_INFO("Object " AssetIdLogStr " creation failed!", AssetIdLogArgs(id));
                                }
                                else {
                                    this->m_cachingStrategy.Add(libraryId, result);

                                    if (!TBuilder::BuildSecondPass(result, context)) {
                                        FEATSTD_LOG_INFO("Object " AssetIdLogStr " build failed!", AssetIdLogArgs(id));
                                        this->Release(id, true);
                                        result = T(0);
                                    }
                                }
                            }
                            this->m_dependencyStack.Remove(this->m_dependencyStack.Size() - 1);
                        }
                    }
                }
            }

            return result;
        }

        template <typename T, typename CachingStrategy, typename TBuilder>
        T TypedCffAssetProvider<T, CachingStrategy, TBuilder>::GetFromCache(AssetId id, DependencyList* dependencyList /* = 0*/)
        {
            CffLoaderContext context = {id, this->m_provider, AssetDataHandle(), dependencyList, false, 0};

#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&this->m_cacheCriticalSection);
            //Object validity might depend on the current theme, so block any concurrent change.
            FeatStd::Internal::CriticalSectionLocker themeLock(&this->m_provider->m_themeChangeCriticalSection);
#endif

            Id libraryId = AssetIdFunctions::GetLibraryId(id);
            T result = this->m_cachingStrategy.Get(libraryId);
            if ((result != 0) && (!TBuilder::IsValid(result, context))) {
                this->m_cachingStrategy.Remove(libraryId);
                result = T(0);
            }
            else {
                // Do nothing.
            }

            return result;
        }

        /**
         *  Non-caching version of TypedCffAssetProvider.
         */
        template <typename T, typename TBuilder = AssetBuilder<T> >
        class CffNonCachingAssetProvider : public TypedCffAssetProvider<T, NoCachingStrategy<T>, TBuilder> {
        public:
            CffNonCachingAssetProvider(DefaultAssetProvider* provider):TypedCffAssetProvider<T, NoCachingStrategy<T>, TBuilder>(provider){}
        };

        /**
         *  Caching version of TypedCffAssetProvider.
         */
        template <typename T, typename TBuilder = AssetBuilder<T> >
        class CffCachingAssetProvider : public TypedCffAssetProvider<T, DefaultCachingStrategy<T>, TBuilder> {
            public:
            CffCachingAssetProvider(DefaultAssetProvider* provider):TypedCffAssetProvider<T, DefaultCachingStrategy<T>, TBuilder>(provider){}
        };
    }
}

#endif  // CANDERA_TypedCffAssetProvider_H
