//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TypedAssetProvider_H)
    #define CANDERA_TypedAssetProvider_H

#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/Generic/TypedAssetProviderBase.h>
#include <CanderaAssetLoader/AssetLoaderBase/Generic/AssetBuilder.h>
#include <CanderaAssetLoader/AssetLoaderBase/Generic/CachingStrategy.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetId.h>

namespace Candera {
    class DefaultAssetProvider;
    namespace Internal {

        /**
         * @brief Generic class for providing assets with caching and 2-pass building mechanisms.
         *
         * When an object is requested, the TypedAssetProvider searches its internal cache, if it is configured to use one.
         *  If the object is found in the cache, it is retrieved from it, otherwise the asset provider
         *  builds the object from the asset lib, places it in the cache and than returns it.
         * The object is build in two passes, one for memory allocation and basic initializations, and
         *  one for initializing properties that might require retrieving other objects via the
         *  AssetProvider interface. The object is cached between the two passes.
         */
        template <typename T, typename CachingStrategy, typename TBuilder = AssetBuilder<T> >
        class TypedAssetProvider : public TypedAssetProviderBase {
            FEATSTD_LOG_SET_REALM(Candera::Diagnostics::LogRealm::CanderaAssetLoader);
            typedef TypedAssetProviderBase Base;

            public:
                class Visitor {
                public:
                    virtual bool Visit(T& value) = 0;
                };

                private:
                class DisposeVisitor: public CachingStrategy::Visitor {
                public:
                    virtual bool Visit(const Id& /*key*/, T& value)
                    {
                        TBuilder::Dispose(value);
                        return true;
                    }
                };

                typedef Visitor InternalVisitor;
                class FilterVisitor : public CachingStrategy::Visitor
                {
                public:
                    virtual bool Visit(const Id& /*key*/, T& value) override
                    {
                        return m_visitor->Visit(value);
                    }
                    InternalVisitor* m_visitor;
                };

            public:
                /**
                 * Construct a TypedAssetProvider object.
                 *
                 * @param provider AssetProvider used to resolve dependencies at object creation time.
                 */
                TypedAssetProvider(DefaultAssetProvider* provider):m_provider(provider)
                {
                }

                /**
                 *  The destructor releases all cached instances, in case of caching is enabled.
                 */
                ~TypedAssetProvider()
                {
                    ReleaseAll();
                }

                /**
                 * Associate an external object with a name.
                 *
                 * This association allows the AssetProvider to bypass object creation for the given name from the asset file,
                 *  returning the provided object instance instead. This set is allowed before any object with this name
                 *  is set or read from the asset file, or after the set/read object is released.
                 *
                 * @param name Name of the object.
                 * @param object Object instance.
                 * @return true if the association succeeded.
                 */
                bool Set(AssetId id, T object)
                {
                    return m_cachingStrategy.Set(AssetIdFunctions::GetLibraryId(id), object);
                }

                /**
                 * Retrieve an object by name.
                 *
                 * @param name Name of the object.
                 * @return Object instance.
                 */
                T Get(AssetId id, DependencyList* dependencyList = 0);

                /**
                 * Retrieve an object by name.
                 *
                 * @param#1 id - Asset's id
                 * @param#2 dependencyList - Default value is null.
                 * @return Object's address or null pointer.
                 */
                T GetFromCache(AssetId id, DependencyList* dependencyList = 0);

                virtual void Release(AssetId id, bool dispose) override;

                void Filter(Visitor& visitor);

                /**
                 * Release all objects.
                 *
                 * The method disposes the objects and it should be called only if the objects are not intended
                 *  to be used anymore
                 */
                void ReleaseAll();

            protected:
                CachingStrategy m_cachingStrategy;
                Candera::Internal::Vector<Id> m_dependencyStack;
                DefaultAssetProvider* m_provider;

#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSection m_cacheCriticalSection;
#endif
            private:
                FEATSTD_MAKE_CLASS_UNCOPYABLE(TypedAssetProvider);
        };

        template <typename T, typename CachingStrategy, typename TBuilder>
        T TypedAssetProvider<T, CachingStrategy, TBuilder>::Get(AssetId id, DependencyList* dependencyList)
        {
            TypedLoaderContext<DirectLoaderType> context = {this->m_provider, id, dependencyList, false};

#ifdef FEATSTD_THREADSAFETY_ENABLED
            //block cache access during object construction
            FeatStd::Internal::CriticalSectionLocker cacheLock(&m_cacheCriticalSection);
            //block theme changing during object construction
            FeatStd::Internal::CriticalSectionLocker themeLock(&this->m_provider->m_themeChangeCriticalSection);
#endif

            Id libraryId = AssetIdFunctions::GetLibraryId(id);
            T result = m_cachingStrategy.Get(libraryId);
            if (!TBuilder::IsValid(result, context)) {
                this->m_cachingStrategy.Remove(libraryId);
                result = T(0);
            }

            if (result == T(0)) {
                if (m_dependencyStack.Contains(libraryId)) {
                    FEATSTD_LOG_ERROR("Cyclic dependency for object " AssetIdLogStr, AssetIdLogArgs(id));
                }
                else {
                    if (!m_dependencyStack.Add(libraryId)) {
                        FEATSTD_LOG_ERROR("Cyclic dependency for object " AssetIdLogStr, AssetIdLogArgs(id));
                    }
                    else {
                        result = TBuilder::CreateAndBuildFirstPass(context);
                        if (result == T(0)) {
                            FEATSTD_LOG_ERROR("Object " AssetIdLogStr " creation failed!", AssetIdLogArgs(id));
                        }
                        else {
                            m_cachingStrategy.Add(libraryId, result);
                            if (!TBuilder::BuildSecondPass(result, context)) {
                                FEATSTD_LOG_ERROR("Object " AssetIdLogStr " build failed!", AssetIdLogArgs(id));
                                Release(id, true);
                                result = T(0);
                            }
                        }

                        m_dependencyStack.Remove(m_dependencyStack.Size() - 1);
                    }
                }
            }

            return result;
        }
        template <typename T, typename CachingStrategy, typename TBuilder>
        T TypedAssetProvider<T, CachingStrategy, TBuilder>::GetFromCache(AssetId id, DependencyList* dependencyList  /* = 0*/)
        {
            TypedLoaderContext<DirectLoaderType> context = {this->m_provider, id, dependencyList, false};

#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_cacheCriticalSection);
            //Object validity might depend on the current theme, so block any concurrent change.
            FeatStd::Internal::CriticalSectionLocker themeLock(&this->m_provider->m_themeChangeCriticalSection);
#endif

            Id libraryId = AssetIdFunctions::GetLibraryId(id);
            T result = m_cachingStrategy.Get(libraryId);
            if (!TBuilder::IsValid(result, context)) {
                this->m_cachingStrategy.Remove(libraryId);
                result = T(0);
            }
            return result;
        }
        template <typename T, typename CachingStrategy, typename TBuilder>
        void TypedAssetProvider<T, CachingStrategy, TBuilder>::Release(AssetId id, bool dispose)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_cacheCriticalSection);
#endif
            T object = m_cachingStrategy.Remove(AssetIdFunctions::GetLibraryId(id));
            if (dispose && object != 0) {
                TBuilder::Dispose(object);
            }
        }
        template <typename T, typename CachingStrategy, typename TBuilder>
        void TypedAssetProvider<T, CachingStrategy, TBuilder>::Filter(Visitor& visitor)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_cacheCriticalSection);
#endif
            FilterVisitor filterVisitor;
            filterVisitor.m_visitor = &visitor;
            m_cachingStrategy.Filter(filterVisitor);
        }
        template <typename T, typename CachingStrategy, typename TBuilder>
        void TypedAssetProvider<T, CachingStrategy, TBuilder>::ReleaseAll()
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_cacheCriticalSection);
#endif
            DisposeVisitor visitor;
            m_cachingStrategy.Visit(visitor);
            m_cachingStrategy.RemoveAll();
        }
        /**
         *  Non-caching version of TypedAssetProvider
         */
        template <typename T>
        class NonCachingAssetProvider : public TypedAssetProvider<T, Candera::Internal::NoCachingStrategy<T>, AssetBuilder<T> > {
            NonCachingAssetProvider(DefaultAssetProvider* provider): TypedAssetProvider<T, Candera::Internal::NoCachingStrategy<T>, AssetBuilder<T> >(provider) {};
        };

        /**
         *  Caching version of TypedAssetProvider
         */
        template <typename T>
        class CachingAssetProvider : public TypedAssetProvider<T, Candera::Internal::DefaultCachingStrategy<T>, AssetBuilder<T> > {
        public:
            CachingAssetProvider(DefaultAssetProvider* provider): TypedAssetProvider<T, Candera::Internal::DefaultCachingStrategy<T>, AssetBuilder<T> >(provider) {};
        };
    }
}

#endif  // CANDERA_TypedAssetProvider_H
