//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CachingStrategy_H)
    #define CANDERA_CachingStrategy_H

#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Container/Map.h>

namespace Candera {
    namespace Internal {

        /**
         *  NoCachingStrategy is used for TypedAssetProvider instances that do not need any caching.
         */
        template <typename T>
        class NoCachingStrategy {
            public:
                typedef typename Map<Id, T>::Visitor Visitor;
                bool Add(Id /*id*/, const T& /*object*/) { return true; }
                bool Set(Id /*id*/, T /*object*/) { return true; }
                T Get(Id /*id*/) { return T(0); }
                bool Visit(Visitor& /*visitor*/) { return true; }
                T Remove(Id /*id*/) { return T(0); }
                void Filter(Visitor& /*visitor*/) {}
                void RemoveAll() {}
        };

        /**
         *  Defines the default caching strategy for TypedAssetProvider.
         */
        template <typename T>
        class DefaultCachingStrategy {
            public:
                DefaultCachingStrategy(): m_cache() {}
                typedef typename Map<Id, T>::Visitor Visitor;
                bool Add(Id id, const T& object) { return m_cache.Insert(id, object); }
                bool Set(Id id, T object) { return m_cache.Insert(id, object); }
                T Get(Id id) { return Get(id, false); }
                bool Visit(Visitor& visitor) { return m_cache.Visit(visitor); }
                T Remove(Id id) { return Get(id, true); }
                void Filter(Visitor& visitor) { return m_cache.Filter(visitor); }
                void RemoveAll() { return m_cache.Clear(); }

            private:
                Map<Id, T> m_cache;

                T Get(Id id, bool remove)
                {
                    T* resultPtr = m_cache.Find(id);
                    if (resultPtr != 0) {
                        T result = *resultPtr;
                        if (remove) {
                            m_cache.Remove(id);
                        }
                        return result;
                    }

                    return T(0);
                }
        };

    }
}   // namespace Candera

#endif  // CANDERA_CachingStrategy_H
