//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSETREADER_H)
#define CANDERA_ASSETREADER_H

#include <Candera/Environment.h>
#include <CanderaAssetLoader/AssetLoaderBase/Generic/AssetLoaderTypeInfo.h>

namespace Candera {
    namespace Internal {

        /**
         * Generic base class for AssetReader specializations.
         *
         * GenericAssetReaderBase provides methods for reading an object of type T in two steps. The
         *  AssetReader specialization can derive this class and hide one or both of the default methods
         *  with its own implementation.
         */
        FEATSTD_LINT_SYMBOL(1577, Candera::Internal::AssetReaderBase<*>, "GenericAssetReaderBase is specialized in each builder")
        FEATSTD_LINT_SYMBOL(1511, Candera::Internal::GenericAssetReaderBase<*>::ReadFirstPass, "Derived classes also provide ReadFirstPass functions")
        FEATSTD_LINT_SYMBOL(1511, Candera::Internal::GenericAssetReaderBase<*>::ReadSecondPass, "Derived classes also provide ReadSecondPass functions")
        template<typename T>
        class GenericAssetReaderBase {
        protected:
            //LoaderContext is one of the specialized LoaderContext<> templates, based on the LoaderType proxy
            // defined with the AssetLoaderTypeInfo macro.
            typedef TypedLoaderContext<static_cast<AssetLoaderType>(AssetLoaderInfo<T>::LoaderType)> LoaderContext;
        public:
            /**
             * ReadFirstPass is the method that is used to initialize the object before it is cached by the
             *  AssetProvider.
             *
             * This method:
             * - MUST set properties that have dependencies to objects of same type, but a cyclic dependency
             *  is not allowed, e.g. the owner of a GraphicDeviceUnit cannot be the same GraphicDeviceUnit. In this
             *  case an error will be signaled by the AssetProvider, since this configuration is an invalid one
             *  and would produce undesired results.
             * - MUST set properties that might be dependencies for other objects, e.g. the scene tree should be
             *  be fully built after completion of this method. This will assure that other objects that might reference
             *  nodes in the scene can get their correct reference, even if not fully constructed.
             * - CAN set properties that do not have dependencies to the same object type, e.g. position of a Node.
             * - MUST NOT set properties that have legal dependencies to objects of the same type, e.g. Camera of
             *  the ReflectionCamera must not be initialized here because the reflected camera might not be
             *  constructed yet. Still, the ReflectionCamera has to be created for the completion of the scene tree.
             *
             * @param object object that has to be initialized.
             * @param context loader context used to initialize the object.
             *
             * @return true if object is successfully initialized, false otherwise.
             */
            static bool ReadFirstPass(T& object, const LoaderContext& context);

            public:
                /**
                 * ReadFirstPass is the method that is used to initialize the object before it is cached by the
                 *  AssetProvider.
                 *
                 * This method:
                 * - MUST set properties that have dependencies to objects of same type, but a cyclic dependency
                 *  is not allowed, e.g. the owner of a GraphicDeviceUnit cannot be the same GraphicDeviceUnit. In this
                 *  case an error will be signaled by the AssetProvider, since this configuration is an invalid one
                 *  and would produce undesired results.
                 * - MUST set properties that might be dependencies for other objects, e.g. the scene tree should be
                 *  be fully built after completion of this method. This will assure that other objects that might reference
                 *  nodes in the scene can get their correct reference, even if not fully constructed.
                 * - CAN set properties that do not have dependencies to the same object type, e.g. position of a Node.
                 * - MUST NOT set properties that have legal dependencies to objects of the same type, e.g. Camera of
                 *  the ReflectionCamera must not be initialized here because the reflected camera might not be
                 *  constructed yet. Still, the ReflectionCamera has to be created for the completion of the scene tree.
                 *
                 * @param object object that has to be initialized.
                 * @param context loader context used to initialize the object.
                 *
                 * @return true if object is successfully initialized, false otherwise.
                 */
                static bool ReadFirstPass(T& object, LoaderContext& context);

                /**
                 * ReadSecondPass is the method that is used to finish the object initialization after it is cached by the
                 *  AssetProvider.
                 *
                 * This method:
                 * - MUST set properties that have legal dependencies to objects of the same type.
                 * - CAN set properties that do not have dependencies to the same object type and were not set by
                 *  the first method.
                 * - MUST NOT set properties that have dependencies to objects of same type, but a cyclic dependency
                 *  is not allowed.
                 * - MUST NOT set properties that might be dependencies for other objects, but
                 *  COULD finalize their initialization, e.g. all children of a Node are created and set in the first pass,
                 *  but in the second pass, they have to fully initialized so for example a RelfectionCamera will get
                 *  the reference to the reflected Camera object.
                 *
                 * @param object object that has to be initialized.
                 * @param context loader context used to initialize the object.
                 *
                 * @return true if object is successfully initialized, false otherwise.
                 */
                static bool ReadSecondPass(T& object, LoaderContext& context);

            protected:
                static bool CreateDependency(LoaderContext& context, T& object, AssetId id, typename TypedDependencyLoader<T>::ReloadFn reloadFn) {
                    if (!id.IsValid()) {
                        return true;
                    }
                    bool createDependency = true;
                    if (!context.iterativeLoad) {
                        createDependency = !reloadFn(object, id, context.reloadList, context.provider);
                    }
                    if (createDependency) {
                        if (context.reloadList != 0) {
                            return context.reloadList->Append(DependencyEntry(ASSETLOADER_TRANSIENT_NEW(TypedDependencyLoader<T>)(object, id, reloadFn, context.provider)));
                        }
                        else {
                            return reloadFn(object, id, context.reloadList, context.provider);
                        }
                    }
                    else {
                        return true;
                    }
                }
        };

        template<typename T>
        inline bool GenericAssetReaderBase<T>::ReadFirstPass(T& /*object*/, LoaderContext& /*context*/)
        {
            return true;
        }

        template<typename T>
        inline bool GenericAssetReaderBase<T>::ReadSecondPass(T& /*object*/, LoaderContext& /*context*/)
        {
            return true;
        }

        /**
         * Default AssetReaderBase implementation that will be used for objects that have not specific
         *  properties that have to be set.
         */
        template<typename T>
        class AssetReaderBase: public GenericAssetReaderBase<T> {};

        /**
         * Generic AssetReader.
         *
         * The generic AssetReader provides methods for reading an object of type T in two steps.
         */
        template<typename T, typename BaseType = typename AssetLoaderInfo<T>::BaseType>
        class AssetReader {
                typedef AssetReader<BaseType> BaseTypeReader;
                typedef TypedLoaderContext<static_cast<AssetLoaderType>(AssetLoaderInfo<T>::LoaderType)> LoaderContext;
            public:
                /**
                 * ReadFirstPass is the method that is used to initialize the object before it is cached by the
                 *  AssetProvider.
                 *
                 * @param object object that has to be initialized.
                 * @param context loader context used to initialize the object.
                 *
                 * @return true if both ReadFirstPass called on the AssetReaderBase specialization
                 *  for type T, and the ReadFirstPass called on the AssetReader for the base type (see AssetLoaderTypeInfo)
                 *  returned true.
                 */
                static bool ReadFirstPass(T& object, LoaderContext& context) {
                    return BaseTypeReader::ReadFirstPass(object, context) && AssetReaderBase<T>::ReadFirstPass(object, context);
                }

                /**
                 * ReadSecondPass is the method that is used to finish the object initialization after it is cached by the
                 *  AssetProvider.
                 *
                 * @param object object that has to be initialized.
                 * @param context loader context used to initialize the object.
                 *
                 * @return true if both ReadSecondPass called on the AssetReaderBase specialization
                 *  for type T, and the ReadSecondPass called on the AssetReader for the base type (see AssetLoaderTypeInfo)
                 *  returned true.
                 */
                static bool ReadSecondPass(T& object, LoaderContext& context) {
                    return BaseTypeReader::ReadSecondPass(object, context) && AssetReaderBase<T>::ReadSecondPass(object, context);
                }
        };

        /**
         * Generic AssetReader specialization for types that have not base type set.
         *
         * The generic AssetReader provides methods for reading an object of type T in two steps.
         */
        template<typename T>
        class AssetReader<T, void> {
                typedef TypedLoaderContext<static_cast<AssetLoaderType>(AssetLoaderInfo<T>::LoaderType)> LoaderContext;
            public:
                /**
                 * ReadFirstPass is the method that is used to initialize the object before it is cached by the
                 *  AssetProvider.
                 *
                 * @param object object that has to be initialized.
                 * @param context loader context used to initialize the object.
                 *
                 * @return true if AssetReaderBase specialization for type T returned true.
                 */
                static bool ReadFirstPass(T& object, LoaderContext& context) {
                    return AssetReaderBase<T>::ReadFirstPass(object, context);
                }

                /**
                 * ReadSecondPass is the method that is used to finish the object initialization after it is cached by the
                 *  AssetProvider.
                 *
                 * @param object object that has to be initialized.
                 * @param context loader context used to initialize the object.
                 *
                 * @return true if AssetReaderBase specialization for type T returned true.
                 */
                static bool ReadSecondPass(T& object, LoaderContext& context) {
                    return AssetReaderBase<T>::ReadSecondPass(object, context);
                }
        };
    }
}

#endif  // CANDERA_ASSETREADER_H
