//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSETLOADERTYPEINFO_H)
#define CANDERA_ASSETLOADERTYPEINFO_H

#include <Candera/Environment.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDataHandle.h>
#include <CanderaAssetLoader/AssetLoaderBase/ReloadEntry.h>

namespace Candera {
    class DefaultAssetProvider;

    namespace Internal {

        /**
         * Specifies the asset loading type.
         */
        enum AssetLoaderType {
            DirectLoaderType,       ///< The object is built based on its name only.
            CffLoaderType,          ///< The object is built from a cff asset file.
            UnknownLoaderType       ///< The load type is unknown.
        };

         template <AssetLoaderType BType = UnknownLoaderType>
         class TypedLoaderContext;

        /**
         * LoaderContext specialization for objects built based on their name.
         *
         * The DirectLoaderContext holds the name of the object to be built.
         */
        template <>
        class TypedLoaderContext<DirectLoaderType> {
        public:
            DefaultAssetProvider* provider;
            AssetId id;
            DependencyList* reloadList;
            bool iterativeLoad;

            bool IsValid() const { return (id.IsValid()); }
        };
        typedef TypedLoaderContext<DirectLoaderType> DirectLoaderContext;
        
        /**
         * LoaderContext specialization for objects built from a cff file.
         *
         * The CffLoaderContext holds the CFFAssetReader instance and the AssetDataHandle corresponding to the
         *  object to be built.
         */
        template <>
        class TypedLoaderContext<CffLoaderType> {
            public:
                AssetId id;
                DefaultAssetProvider* provider;
                AssetDataHandle handle;
                DependencyList* reloadList;
                bool iterativeLoad;
                UInt16 repositoryId;

                bool IsValid() const { return (handle.IsValid()); }
                TypedLoaderContext Clone(const AssetDataHandle& newHandle) const
                {
                    TypedLoaderContext result = *this;
                    result.handle = newHandle;
                    return result;
                }
        };
        typedef TypedLoaderContext<CffLoaderType> CffLoaderContext;

        /**
         * Proxy for defining asset objects inheritance tree and asset loading type
         */
        template <typename T>
        struct AssetLoaderInfo {
            typedef void BaseType;
            enum { LoaderType = UnknownLoaderType};
        };

        /**
         * Macro for AssetLoaderInfo proxy expansion.
         *
         * @param Class         Type of object for which the AssetLoaderInfo is defined, e.g.Transformable.
         * @param LoaderTypeP   AssetLoaderType enum that defines how the 'Class' will be built - from file or by name.
         * @param BaseClass     Base type of the 'Class'. The BaseClass defines which type of base class will be built
         *          along with a Class object type, e.g. for Transformable the base class would be CanderaObject. The
         *          BaseClass is not necessary the Candera defined base class, but rather the one that is defined
         *          by the cff format.
         */
        FEATSTD_LINT_SYMBOL(1577, Candera::Internal::AssetLoaderInfo<*>, "AssetLoaderInfo is specialized in each builder")
    #define AssetLoaderTypeInfo(Class, LoaderTypeP, BaseClass)                      \
        template <> struct AssetLoaderInfo<Class>                                   \
        {                                                                           \
            typedef BaseClass BaseType;                                             \
            enum { LoaderType = LoaderTypeP };                                      \
        }

    #define AssetLoaderTypeInfoWithNamespace(Namespace, Class, LoaderTypeP, BaseClass)  \
        template <> struct AssetLoaderInfo<Namespace::Class>                            \
        {                                                                               \
        typedef BaseClass BaseType;                                                     \
        enum { LoaderType = LoaderTypeP };                                              \
        }
    }
}

#endif  // CANDERA_ASSETLOADERTYPEINFO
