//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "DefaultResourceProvider.h"

#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetGroup.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetCache.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ItemHeaderCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDecompression.h>
#include <Candera/EngineBase/Common/ResourceObject.h>
#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    namespace Internal {

#ifdef FEATSTD_THREADSAFETY_ENABLED
        static FeatStd::Internal::CriticalSection* GetCacheAccessCriticalSection()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, s_assetCacheCriticalSection);
            return &s_assetCacheCriticalSection;
        }
#endif

        using namespace Diagnostics;

        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        DefaultResourceProvider& DefaultResourceProvider::GetResourceProviderInstance()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(DefaultResourceProvider, resourceProvider);
            return resourceProvider;
        }

        ResourceDataHandle DefaultResourceProvider::CreateRegionDataHandle(AssetLib libType, Id id, UInt16 regionId, UInt32 size)
        {
            if (m_assetGroup == 0) {
                FEATSTD_LOG_WARN("RegionDataHandle cannot be created. Inconsistent state reached, "
                    "seeming that DefaultAssetProvider is not initialized.");
                return ResourceDataHandle::InvalidHandle();
            }
            const AssetId& assetId = AssetIdFunctions::CreateId(libType, AotUnknown, id, 0);
            ResourceDataHandle result = m_assetGroup->GetItemHeaderHandle(assetId);
            if (!result.m_accessor.IsValid()) {
                FEATSTD_LOG_WARN("RegionDataHandle cannot be created. Header information for "
                    "library with AssetId: " AssetIdLogStr " cannot be obtained from the asset",
                    AssetIdLogArgs(assetId));
            }
            else {
                result.m_size = size;
                result.m_accessor.m_asset.m_regionIndex = regionId;
            }

            return result;
        }

        const void* DefaultResourceProvider::Lock(const ResourceDataHandle& resourceDataHandle, bool& isPersistent)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(GetCacheAccessCriticalSection());
#endif
            if (m_assetGroup == 0) {
                FEATSTD_LOG_WARN("Resource cannot be locked. DefaultAssetProvider must be initialized "
                    "for resource access");
                return 0;
            }

            const AssetCache::Key& key = resourceDataHandle.m_accessor.m_asset;

            if (isPersistent == false) {
                const void* result = AssetCache::GetInstance().GetAndRetain(key);
                if (result != 0) {
                    return result;
                }
            }

            AssetSet* assetSet = m_assetGroup->GetAssetSet(key.m_repositoryId);
            if (assetSet == 0) {
                return 0;
            }

            UInt32 uncompressedSize = 0;

            ResourceDataHandle handle = resourceDataHandle;
            if (handle.m_accessor.m_asset.m_regionIndex != NOT_A_REGION) {
                AssetDataHandle headerHandle(assetSet->CreateAssetItemHeaderHandle(handle.m_accessor.m_asset.m_offset));
                if (!headerHandle.IsValid()) {
                    return 0;
                }
                const AssetDataHandle& regionHandle = CFFReader::GetRegion(headerHandle, handle.m_accessor.m_asset.m_regionIndex);
                UInt32 regionOffset = CFFReader::GetRegionDataOffset(regionHandle);
                if (regionOffset < CFF_INT_SIZE) {
                    return 0;
                }
                regionOffset -= CFF_INT_SIZE;

                uncompressedSize = static_cast<UInt32>(CFFReader::GetRegionDataUncompressedSize(regionHandle));
                if (uncompressedSize == 0) {
                    if (regionHandle.IsPersistent()) {
                        isPersistent = true;
                        return headerHandle.GetData(regionOffset);
                    }
                    else {
                        if (isPersistent) {
                            return 0;
                        }
                    }
                }
                else {
                    if (isPersistent) {
                        return 0;
                    }
                }
                handle.m_accessor.m_asset.m_offset += regionOffset;
                handle.m_size = CFFReader::GetRegionDataSize(regionHandle);
                handle.m_accessor.m_asset.m_regionIndex = NOT_A_REGION;
            }
            else {
                const void* data = assetSet->GetAddress(key.m_offset, resourceDataHandle.m_size);
                if (data != 0) {
                    isPersistent = true;
                    return data;
                }
                else {
                    if (isPersistent) {
                        return 0;
                    }
                }
            }

            void* newResult = AssetCache::GetInstance().Reserve(key, resourceDataHandle.m_size);
            if (newResult == 0) {
                return 0;
            }

            if (uncompressedSize != 0) {
                if (uncompressedSize != AssetDecompressor::UncompressData(handle, uncompressedSize, newResult)) {
                    AssetCache::GetInstance().Release(newResult);
                    newResult = 0;
                }
            }
            else {
                if (handle.m_size != assetSet->ReadData(newResult, handle.m_accessor.m_asset.m_offset, handle.m_size)) {
                    AssetCache::GetInstance().Release(newResult);
                    newResult = 0;
                }
            }

            return newResult;
        }

        void DefaultResourceProvider::Retain(const void* data)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(GetCacheAccessCriticalSection());
#endif
            AssetCache::GetInstance().Retain(data);
        }

        void DefaultResourceProvider::Release(const void* data)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(GetCacheAccessCriticalSection());
#endif
            AssetCache::GetInstance().Release(data);
        }

        SizeType DefaultResourceProvider::ReadData(void* buffer, const ResourceDataHandle& handle, SizeType offset, SizeType count)
        {
            if (m_assetGroup == 0) {
                return 0;
            }

            AssetSet* assetSet = m_assetGroup->GetAssetSet(handle.m_accessor.m_asset.m_repositoryId);
            if (assetSet == 0) {
                return 0;
            }

            Int32 uncompressedSize = 0;

            ResourceDataHandle newHandle = handle;
            if (newHandle.m_accessor.m_asset.m_regionIndex != NOT_A_REGION) {
                AssetDataHandle headerHandle(assetSet->CreateAssetItemHeaderHandle(newHandle.m_accessor.m_asset.m_offset));
                if (!headerHandle.IsValid()) {
                    return 0;
                }
                const AssetDataHandle& regionHandle = CFFReader::GetRegion(headerHandle, newHandle.m_accessor.m_asset.m_regionIndex);
                // Store number as an signed to be safe for later substract operation
                UInt32 regionOffset = CFFReader::GetRegionDataOffset(regionHandle);
                
                if (regionOffset <= CFF_INT_SIZE) {
                    return 0;
                } 
                regionOffset -= CFF_INT_SIZE;

                uncompressedSize = CFFReader::GetRegionDataUncompressedSize(regionHandle);
                if (uncompressedSize == 0) {
                    if (regionHandle.IsPersistent()) {
                        MemoryPlatform::Copy(buffer, headerHandle.GetData(static_cast<OffsetType>(regionOffset + offset)), count);
                        return count;
                    }
                }
                newHandle.m_accessor.m_asset.m_offset += regionOffset;
                newHandle.m_size = CFFReader::GetRegionDataSize(regionHandle);
                newHandle.m_accessor.m_asset.m_regionIndex = NOT_A_REGION;
            }

            if (uncompressedSize != 0) {
                if ((FeatStd::Internal::NumericConversion<SizeType, Int32>(uncompressedSize) != count) || (0 != offset))
                {
                    FEATSTD_LOG_FATAL("Tried to read data in partitions or with an offset from a compressed resource.");
                    return 0;
                }
                if (uncompressedSize != static_cast<Int32>(AssetDecompressor::UncompressData(newHandle, uncompressedSize, buffer))) {
                    return 0;
                }
                return count;
            }
            else {
                return assetSet->ReadData(buffer, static_cast<OffsetType>(newHandle.m_accessor.m_asset.m_offset + offset), count);
            }
        }

    }   // namespace Internal
}   // namespace Candera
