//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_COMPOSITEGROUPFINDER_H)
#define CANDERA_COMPOSITEGROUPFINDER_H

#include <Candera/Environment.h>

#ifdef CANDERA_3D_ENABLED
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <Candera/Engine3D/Core/CompositeGroup.h>
#endif
#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/TreeTraverser2D.h>
#include <Candera/Engine2D/Core/CompositeGroup2D.h>
#endif

namespace Candera {
    namespace Internal {
        //it is assumed that the root passed at constructor is the same that is the one for which Traverse is called

        template<typename TNode, typename TCompositeGroup>
        class CompositeGroupFinder : public TreeTraverserBase<TNode>
        {
            typedef ::Candera::TreeTraverserBase<TNode> Base;
            typedef typename Base::TraverserAction TraverserAction;
            typedef typename TCompositeGroup::AnchorPointIterator AnchorPointIterator;
        public:
            CompositeGroupFinder(TNode* root, ::Candera::Id nodeId)
                :m_rootCompositeGroup(::Candera::Dynamic_Cast<TCompositeGroup*>(root)),
                m_nodeId(nodeId),
                m_result(0)
            {
            }

            TNode* GetResult() const
            {
                return m_result;
            }

            static TCompositeGroup* GetCompositeGroup(TNode* root, Int32 pathLength, const UInt32* const * assetIdChain)
            {
                const UInt32* const actualAssetIdChain = reinterpret_cast<const UInt32* >(assetIdChain);
                TNode* result = root;
                for (Int32 nodeIndex = 0; ((nodeIndex < pathLength) && (result != 0)); ++nodeIndex) {
                    const AssetId& compositeGroupId = AssetIdFunctions::GetAssetId(actualAssetIdChain + 4 * nodeIndex);
                    if (compositeGroupId.IsValid()) {
                        CompositeGroupFinder<TNode, TCompositeGroup> compositeGroupFinder(result, AssetIdFunctions::GetNodeId(compositeGroupId));
                        compositeGroupFinder.Traverse(*result);
                        result = compositeGroupFinder.GetResult();
                    }
                    else {
                        result = 0;
                    }
                }

                return Dynamic_Cast<TCompositeGroup*>(result);
            }

        protected:
            virtual TraverserAction ProcessNode(TNode& node) override
            {
                if (node.GetId() == m_nodeId) {
                    m_result = &node;
                    return Base::StopTraversing;
                }

                if (m_rootCompositeGroup != 0) {
                    for (AnchorPointIterator iterator = m_rootCompositeGroup->GetAnchorPointIterator(); iterator.IsValid(); ++iterator) {
                        if (&node == iterator->m_node) {
                            return Base::StopTraversingForDescendants;
                        }
                    }
                }

                TCompositeGroup* currentCompositeGroup = ::Candera::Dynamic_Cast<TCompositeGroup*>(&node);
                if ((currentCompositeGroup != 0) && (currentCompositeGroup != m_rootCompositeGroup)) {
                    for (AnchorPointIterator iterator = currentCompositeGroup->GetAnchorPointIterator(); iterator.IsValid(); ++iterator) {
                        if (iterator->m_node != 0) {
                            CompositeGroupFinder inAnchorFinder(iterator->m_node, m_nodeId);
                            inAnchorFinder.Traverse(*iterator->m_node);
                            m_result = inAnchorFinder.GetResult();
                            if (m_result != 0) {
                                return Base::StopTraversing;
                            }
                        }
                    }
                    return Base::StopTraversingForDescendants;
                }

                return Base::ProceedTraversing;
            }
        private:
            TCompositeGroup* m_rootCompositeGroup;
            ::Candera::Id m_nodeId;
            TNode* m_result;
        };
    }
}

#endif  // CANDERA_COMPOSITEGROUPFINDER_H
