//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CFFASSETREADER_H)
#define CANDERA_CFFASSETREADER_H

#include <Candera/Environment.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDataHandle.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetId.h>

namespace Candera {

    namespace Internal {

        typedef unsigned char CFF_BYTE;

        static const UInt32 CFF_CHAR_SIZE         = 1U;
        static const UInt32 CFF_STRINGLENGTH_SIZE = 2U;
        static const UInt32 CFF_INT16_SIZE        = 4U;
        static const UInt32 CFF_UINT16_SIZE       = 4U;
        static const UInt32 CFF_FLOAT_SIZE        = 4U;
        static const UInt32 CFF_BOOL_SIZE         = 4U;
        static const UInt32 CFF_INT_SIZE          = 4U;
        static const UInt32 CFF_UINT_SIZE         = 4U;
        static const UInt32 CFF_UINT64_SIZE       = 8U;
        static const UInt32 CFF_INT64_SIZE        = 8U;
        static const UInt32 CFF_DOUBLE_SIZE       = 8U;
        static const UInt32 CFF_PTR_SIZE          = 4U;

        /**
         * @brief AssetReader for new CompactFileFormat asset format
         *
         * CompactFormat defines an asset format designed for fast access to any component
         * of any asset at any time. Its structure:
         *  - CompactHeader: contains asset general description, like version, timestamp etc and
         *     offsets to beginning of each AssetLib (see below)
         *  - NameTable: list of commonly used strings across the libs. Everywhere a string
         *     must be used in an asset, instead of a char array, there is a pointer to a
         *     location in NameTable where the char array can be found
         *  - A group of AssetLibs. Each AssetLib is a compact set of assets of the same type
         *     (scene, animation, resource etc). Each AssetLib is composed of the following:
         *     - AssetLibHeader (data size, asset count etc)
         *     - AssetLibLookUpTree (balanced binary search tree of asset access information In AssetLib)
         *     - A collection of Assets each having its own structure depending on type
         *
         * CFFAssetReader must be initialized with a filePath where the CompactFileFormat libs can be found.
         * CFFAssetReader provide methods for asset opening, data querying (asset type dependent) and closing.
         * For getting a specific data from an asset, after initialization, OpenData for the corresponding
         * AssetLib must be called. This method will return a handle used latter in association with a correct
         * getter. After all the data querying has occurred, CloseData must be called for the AssetLib handle.
         *
         * CFFAssetReader permits handling only one asset from each AssetLib at one moment. For example
         * a Mesh and a VertexBuffer can be opened at the same time, but not two meshes nor two VertexBuffers
         */
        namespace CFFReader {
                static const UInt32 InvalidVal = 0xFFffFFff;
                static const UInt32 CFF_ITEMID_SIZE = 4U * CFF_INT_SIZE;
                static const UInt32 CFF_COLORMASK_SIZE = 4U * CFF_FLOAT_SIZE;
                static const UInt32 CFF_COLOR_SIZE  = 4U * CFF_FLOAT_SIZE;
                static const UInt32 CFF_RECTANGLE_SIZE = 4U * CFF_FLOAT_SIZE;
                static const UInt32 CFF_VECTOR3_SIZE = 3U * CFF_FLOAT_SIZE;
                static const UInt32 CFF_VECTOR2_SIZE = 2U * CFF_FLOAT_SIZE;
                static const UInt32 CFF_MATRIX4_SIZE = 16U * CFF_FLOAT_SIZE;

                inline AssetDataHandle CffGetHandle(const AssetDataHandle& handle, Int32 offset)
                {
                    return (offset == -1) ? AssetDataHandle() : (handle + offset);
                }

                inline AssetDataHandle CffGetAssetLibHandle(const AssetDataHandle& handle)
                {
                    return AssetDataHandle(handle.m_resourceObject, 0);
                }

                template <typename T>
                inline const T* CffGetPtrSafe(const AssetDataHandle& handle, UInt32 offset)
                {
                    return FeatStd::Internal::PointerToPointer<const T*>(handle.GetData(offset));
                }

                template <typename T>
                inline T CffGetVal(const AssetDataHandle& handle, UInt32 offset)
                {
                    return (offset == InvalidVal) ? T(0) : *CffGetPtrSafe<T>(handle, offset);
                }

                template <typename T>
                inline T CffGetValSafe(const AssetDataHandle& handle, UInt32 offset)
                {
                    return *CffGetPtrSafe<T>(handle, offset);
                }

                template <typename T>
                inline const T* CffGetPtr(const AssetDataHandle& handle, UInt32 offset)
                {
                    return (offset == InvalidVal) ? 0 : CffGetPtrSafe<T>(handle, offset);
                }

                //TODO generate from SC:
                // ----------------------------- Shader Params Semantic list -------------------------------------------------------------------

                static const UInt32 CFF_SHADERPARAMS_SEMANTICS_SIZE_OFFSET       = 0U;
                static const UInt32 CFF_SHADERPARAMS_SEMANTICS_COUNT_OFFSET      = 4U  + CFF_SHADERPARAMS_SEMANTICS_SIZE_OFFSET;
                static const UInt32 CFF_SHADERPARAMS_SEMANTICS_LIST_PTR_OFFSET   = 4U  + CFF_SHADERPARAMS_SEMANTICS_COUNT_OFFSET;

                inline Int32 GetShaderParamsCount(const AssetDataHandle& handle)
                { return CffGetValSafe<Int32>(handle, CFF_SHADERPARAMS_SEMANTICS_COUNT_OFFSET);}

                inline Int32 GetShaderParamSemantic(const AssetDataHandle& handle, UInt32 index)
                {
                    const AssetDataHandle& spsListHandle = CffGetHandle(handle, CFF_SHADERPARAMS_SEMANTICS_LIST_PTR_OFFSET);
                    return CffGetValSafe<Int32>(spsListHandle, (CFF_INT_SIZE + CFF_PTR_SIZE) * index);
                }

                inline UInt32 GetShaderParamName(const AssetDataHandle& handle, UInt32 index)
                {
                    const AssetDataHandle& spsListHandle = CffGetHandle(handle, CFF_SHADERPARAMS_SEMANTICS_LIST_PTR_OFFSET);
                    return CffGetValSafe<UInt32>(spsListHandle, CFF_INT_SIZE + ((CFF_INT_SIZE + CFF_PTR_SIZE) * index));
                }

        }
    }   // namespace Internal
}   // namespace Candera

#endif // CANDERA_CFFASSETREADER_H
