//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASYNCASSETPROVIDERPROXY_H)
#define CANDERA_ASYNCASSETPROVIDERPROXY_H

#include <Candera/Environment.h>
#include <Candera/System/EntityComponentSystem/Entity.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetProviderDispatcher.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

namespace FeatStd {
    template <typename _ResultType>
    class AsyncRequest;
}

namespace Candera {
    namespace Animation {
        class AnimationPlayerBase;
    }

    namespace TextRendering {
        class SharedStyle;
    }

    namespace Globalization {
        class LanguagePack;
    }

    class Bitmap;
    class SceneContext;
    class Scene2DContext;
    class GraphicDeviceUnit;
    class CameraGroup;
    class AssetProvider;
    class BitmapImage2D;
    class Shader;
    class VertexBuffer;
    class Appearance;
    class Material;
    class RenderMode;
    class AbstractShaderParamSetter;
    class Texture;
    class BitmapTextureImage;

/** @addtogroup AssetLoaderBase
 *  @{
 */

    /**
     * Proxy for asynchronously loading assets through the AssetProvider interface.
     *
     * Each Get* (GetScene) function schedules a Get* (GetSceneById) call on the AssetProvider that created the Proxy.
     * In the case that CANDERA_ASSETLOADER_WROKER_THREAD_ENABLED flag is enabled in CMAKE, the AssetProvider::Get* call
     * will be dispatched on a separate thread and the result will be available as soon as the worker thread completes
     * the dispatch. In the case that the above mentioned cmake option is disabled, one or more calls to DispatchNext()
     * need to be performed on the current thread to assure that the request is dispached. It is advised to call the 
     * DispatchNext() routine periodically during the Update part of the render loop of the application to ensure all
     * asyncronous asset load requests are dispatched in a controlled and limitted time.
     */
    class AsyncAssetProviderProxy: public Candera::EntityComponentSystem::Entity
    {
        friend class AssetProvider;
        friend class ContentLoader;

    public:
        typedef FeatStd::AsyncRequest<SceneContext*> SceneContextAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<SceneContextAsyncRequest> SceneContextAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<Scene2DContext*> Scene2DContextAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<Scene2DContextAsyncRequest> Scene2DContextAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<Bitmap> > BitmapAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<BitmapAsyncRequest> BitmapAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> > AnimationAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<AnimationAsyncRequest> AnimationAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<GraphicDeviceUnit*> GraphicDeviceUnitAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<GraphicDeviceUnitAsyncRequest> GraphicDeviceUnitAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<TextRendering::SharedStyle> > TextStyleAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<TextStyleAsyncRequest> TextStyleAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<CameraGroup*> CameraGroupAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<CameraGroupAsyncRequest> CameraGroupAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<Globalization::LanguagePack> > LanguagePackAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<LanguagePackAsyncRequest> LanguagePackAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<BitmapImage2D> > BitmapImage2DAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<BitmapImage2DAsyncRequest> BitmapImage2DAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<Shader> > ShaderAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<ShaderAsyncRequest> ShaderAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<VertexBuffer> > VertexBufferAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<VertexBufferAsyncRequest> VertexBufferAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<Appearance> > AppearanceAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<AppearanceAsyncRequest> AppearanceAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<Material> > MaterialAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<MaterialAsyncRequest> MaterialAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<RenderMode> > RenderModeAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<RenderModeAsyncRequest> RenderModeAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<AbstractShaderParamSetter> > AbstractShaderParamSetterAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<AbstractShaderParamSetterAsyncRequest> AbstractShaderParamSetterAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<Texture> > TextureAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<TextureAsyncRequest> TextureAsyncRequestSharedPointer;

        typedef FeatStd::AsyncRequest<FeatStd::MemoryManagement::SharedPointer<BitmapTextureImage> > BitmapTextureImageAsyncRequest;
        typedef FeatStd::MemoryManagement::SharedPointer<BitmapTextureImageAsyncRequest> BitmapTextureImageAsyncRequestSharedPointer;

#ifdef CANDERA_3D_ENABLED
        /**
         * Asynchronous version of AssetProvider::GetSceneById.
         * Returns an AsyncRequest<SceneContext*>::SharedPointer object.
         */
        SceneContextAsyncRequestSharedPointer GetScene(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetShader
          * Returns a AsyncRequest<Shader::SharedPointer>::SharedPointer object.
          */
         ShaderAsyncRequestSharedPointer GetShader(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetVertexBufferById
          * Returns a AsyncRequest<VertexBuffer::SharedPointer>::SharedPointer object.
          */
         VertexBufferAsyncRequestSharedPointer GetVertexBuffer(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetAppearanceById
          * Returns a AsyncRequest<Appearance::SharedPointer>::SharedPointer object.
          */
         AppearanceAsyncRequestSharedPointer GetAppearance(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetMaterialById
          * Returns a AsyncRequest<Material::SharedPointer>::SharedPointer object.
          */
         MaterialAsyncRequestSharedPointer GetMaterial(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetRenderModeById
          * Returns a AsyncRequest<RenderMode::SharedPointer>::SharedPointer object.
          */
         RenderModeAsyncRequestSharedPointer GetRenderMode(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetShaderParamSetterById
          * Returns a AsyncRequest<AbstractShaderParamSetter::SharedPointer>::SharedPointer object.
          */
         AbstractShaderParamSetterAsyncRequestSharedPointer GetShaderParamSetter(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetTextureById
          * Returns a AsyncRequest<Texture::SharedPointer>::SharedPointer object.
          */
         TextureAsyncRequestSharedPointer GetTexture(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetBitmapTextureImageById
          * Returns a AsyncRequest<BitmapTextureImage::SharedPointer>::SharedPointer object.
          */
         BitmapTextureImageAsyncRequestSharedPointer GetBitmapTextureImage(Id id, bool waitForComplete = false);
#endif

#ifdef CANDERA_2D_ENABLED
        /**
         * Asynchronous version of AssetProvider::GetScene2DById.
         * Returns a AsyncRequest<Scene2DContext*>::SharedPointer object.
         */
         Scene2DContextAsyncRequestSharedPointer GetScene2D(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetBitmapImage2DById
          * Returns a AsyncRequest<BitmapImage2D::SharedPointer>::SharedPointer object.
          */
         BitmapImage2DAsyncRequestSharedPointer GetBitmapImage2D(Id id, bool waitForComplete = false);
#endif

        /**
         * Asynchronous version of AssetProvider::GetBitmapById
         * Returns a AsyncRequest<Bitmap::SharedPointer>::SharedPointer object.
         */
         BitmapAsyncRequestSharedPointer GetBitmap(Id id, bool waitForComplete = false);
        
        /**
         * Asynchronous version of AssetProvider::GetAnimationById
         * Returns a AsyncRequest<AnimationPlayerBase::SharedPointer>::SharedPointer object.
         */
         AnimationAsyncRequestSharedPointer GetAnimation(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetGraphicDeviceUnitById
          * Returns a AsyncRequest<GraphicDeviceUnit*> object.
          */
         GraphicDeviceUnitAsyncRequestSharedPointer GetGraphicDeviceUnit(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetTextStyleById
          * Returns a AsyncRequest<SharedStyle::SharedPointer>::SharedPointer object.
          */
         TextStyleAsyncRequestSharedPointer GetTextStyle(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetCameraGroupById
          * Returns a AsyncRequest<GraphicDeviceUnit*> object.
          */
         CameraGroupAsyncRequestSharedPointer GetCameraGroup(Id id, bool waitForComplete = false);

         /**
          * Asynchronous version of AssetProvider::GetLanguagePackById
          * Returns a AsyncRequest<LanguagePack::SharedPointer>::SharedPointer object.
          */
         LanguagePackAsyncRequestSharedPointer GetLanguagePack(Id id, bool waitForComplete = false);

        /**
         * Dispatch next asynchronous request.
         *
         * This function needs to be called periodically to ensure that active asyncronous requests are resolved.
         * One option to call DispatchNext would be to introduce it in the LateUpdate part of the UpdateSystem
         * provided by Candera.
         *
         * @code
         * AssetProvider* assetProvider = ContentLoader::GetInstance().GetAssetProvider();
         * if (assetProvider != 0) {
         *   UpdateSystem* updateSystem = EntityComponentSystem::EntitySystem::Get<UpdateSystem>();
         *   if (updateSystem != 0) {
         *     UpdateSystem::Handle updateHandle = updateSystem->CreateComponent();
         *     UpdateSystem::Delegate delegate = UpdateSystem::Delegate::Update<AsyncAssetProviderProxy, &AsyncAssetProviderProxy::DispatchNext>();
         *     updateSystem->SetComponentLateUpdateDelegate(updateHandle, delegate);
         *     updateSystem->AttachComponent(updateHandle, &assetProvider->AsyncProxy());
         *   }
         * }
         * @endcode
         */
        void DispatchNext();

        /**
         * Get the worker thread for the AssetProviderDispatcher. 
         * @return Pointer to worker thread or \em null.
         */
        FeatStd::AsyncRequestDispatcherWorkerThread *GetAssetProviderDispatcherWorkerThread() { return m_dispatcher.GetWorkerThread(); }

    private:
        AsyncAssetProviderProxy(AssetProvider& assetProvider);
        ~AsyncAssetProviderProxy();

        FEATSTD_MAKE_CLASS_UNCOPYABLE(AsyncAssetProviderProxy);

        AssetProvider& m_assetProvider;
#ifdef CANDERA_ASSETLOADER_WORKER_THREAD_ENABLED
        Internal::ThreadedAssetProviderDispatcher m_dispatcher;
#else
        Internal::AsyncAssetProviderDispatcher m_dispatcher;
#endif
    };

  
 /** @} */ // end of AssetLoaderBase
}
#endif // CANDERA_ASSETPROVIDER_H
