//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AsyncAssetProviderProxy.h"

#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>
#include <FeatStd/Async/AsyncRequest.h>

#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/EngineBase/Animation/AnimationPlayerBase.h>

#ifdef CANDERA_3D_ENABLED
#include <CanderaAssetLoader/AssetLoader3D/SceneContext.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/Engine3D/Core/Appearance.h>
#include <Candera/Engine3D/Core/Material.h>
#include <Candera/Engine3D/Core/RenderMode.h>
#include <Candera/Engine3D/Core/Texture.h>
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
#include <CanderaAssetLoader/AssetLoader2D/Scene2DContext.h>
#include <Candera/Engine2D/Core/BitmapImage2D.h>
#endif //CANDERA_2D_ENABLED

#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <Candera/TextEngine/Style.h>

namespace Candera {

    using namespace Diagnostics;
    using namespace Internal;
    
    template <typename Type>
    struct AssetGetter
    {
        typedef typename FeatStd::AsyncRequest<Type>::SharedPointer Result;
        typedef Type(AssetProvider::*Callback)(Id);

        static Type AssetProviderHelperGetter(AssetProvider& provider, Callback callback, Id id)
        {
            return (provider.*callback)(id);
        }

        static Result GetAsset(FeatStd::AsyncRequestDispatcher& dispatcher, AssetProvider& assetProvider, Callback callback, Id assetId, bool waitForComplete)
        {
            typedef FeatStd::Internal::ResultCallbackAsyncRequest3<Type, AssetProvider&, Callback, Id> LocalType;
            Result result(FEATSTD_NEW(LocalType)(&AssetProviderHelperGetter, assetProvider, callback, assetId));

            UInt32 priority = waitForComplete ? 1 : 0;

            if (result != 0) {
                dispatcher.Schedule(result, priority);
                if (waitForComplete) {
                    FEATSTD_LINT_IGNORE_RETURN_VALUE(dispatcher.WaitForNextExecuteFinished());
                }
            }

            return result;
        }
    };

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

    AsyncAssetProviderProxy::AsyncAssetProviderProxy(AssetProvider& assetProvider):
        m_assetProvider(assetProvider)
    {
    }

    AsyncAssetProviderProxy::~AsyncAssetProviderProxy()
    {
    }

    FEATSTD_LINT_NONCONST_METHOD(Candera::AsyncAssetProviderProxy::DispatchNext, "constness dependes on cmake flags")
    void AsyncAssetProviderProxy::DispatchNext()
    {
#ifndef CANDERA_ASSETLOADER_WORKER_THREAD_ENABLED
        m_dispatcher.GetRequestDispatcher().DispatchNext(false);
#endif
    }

#define GetterDefinition(ResultType, GetterName, AssetProviderGetterName) \
    FeatStd::AsyncRequest<ResultType >::SharedPointer AsyncAssetProviderProxy::GetterName(Id id, bool waitForComplete) \
    { \
        return AssetGetter<ResultType>::GetAsset(m_dispatcher.GetRequestDispatcher(), m_assetProvider, &AssetProvider::AssetProviderGetterName, id, waitForComplete); \
    }

#ifdef CANDERA_3D_ENABLED
    GetterDefinition(SceneContext*, GetScene, GetSceneById)
    GetterDefinition(MemoryManagement::SharedPointer<Shader>, GetShader, GetShader)
    GetterDefinition(VertexBuffer::SharedPointer, GetVertexBuffer, GetVertexBufferById)
    GetterDefinition(Appearance::SharedPointer, GetAppearance, GetAppearanceById)
    GetterDefinition(Material::SharedPointer, GetMaterial, GetMaterialById)
    GetterDefinition(RenderMode::SharedPointer, GetRenderMode, GetRenderModeById)
    GetterDefinition(AbstractShaderParamSetter::SharedPointer, GetShaderParamSetter, GetShaderParamSetterById)
    GetterDefinition(Texture::SharedPointer, GetTexture, GetTextureById)
    GetterDefinition(BitmapTextureImage::SharedPointer, GetBitmapTextureImage, GetBitmapTextureImageById)
#endif

#ifdef CANDERA_2D_ENABLED
    GetterDefinition(Scene2DContext*, GetScene2D, GetScene2DById)
    GetterDefinition(BitmapImage2D::SharedPointer, GetBitmapImage2D, GetBitmapImage2DById)
#endif

    GetterDefinition(Bitmap::SharedPointer, GetBitmap, GetBitmapById)
    GetterDefinition(Animation::AnimationPlayerBase::SharedPointer, GetAnimation, GetAnimationById)
    GetterDefinition(GraphicDeviceUnit*, GetGraphicDeviceUnit, GetGraphicDeviceUnitById)
    GetterDefinition(TextRendering::SharedStyle::SharedPointer, GetTextStyle, GetTextStyleById)
    GetterDefinition(CameraGroup*, GetCameraGroup, GetCameraGroupById)
    GetterDefinition(Globalization::LanguagePack::SharedPointer, GetLanguagePack, GetLanguagePackById)

}
