//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSET_VALIDATION_H)
    #define CANDERA_ASSET_VALIDATION_H

#include <CanderaAssetLoader/AssetLoaderBase/AssetLibHeadInformation.h>
#include <Candera/Environment.h>

namespace Candera {

    namespace Internal {
        class AssetSet;
    }
    class AssetRepository;

/** @addtogroup AssetLoaderBase
 *  @{
 */

    /**
     * CFF asset attributes that can be validated at asset load time. Asset attributes must be used
     *  in conjunction with AssetValidationLevel flags to define overall validation of the asset.
     */
    enum AssetValidationAttributes {
        CffVersionAttribute,                    ///<CFF version - Candera::AssetValidation::CURRENT_VERSION - must match to assure all asset items are loaded correctly.
        WidgetSetVersionAttribute,              ///<WidgetSet version - Candera::MetaInfo::WidgetSetMetaInfo::GetVersion - should be checked if WidgetSet is versioned.
        WidgetSetHashAttribute,                 ///<WidgetSet hash - Candera::MetaInfo::MetaInfoBase::GetHash - should be checked to assure widget semantics (widget names, property names and property types) match one to one with the WidgetSet used to generate the asset.
        CanderaVersionAttribute,                ///<CanderaVersion - CanderaVersionGen.h - should be checked to assure the widget was generated for the same Candera version as the current one.
        FractionalNumberRepresentationAttribute,///<Fractional number representation must be check to assure that the Float representation written in the asset is supported by the platform (see FEATSTD_FRACTIONAL_NUMBER_TYPE_NAME).
        PlatformNameAttribute,                  ///<Platform name - Candera::DevicePackageDescriptor::GetGraphicPackageName - should be checked, especially on target builds, to assure that the asset was generated for the current platform.
        PlatformInstanceNameAttribute,          ///<Platform instance name - Candera::DevicePackageDescriptor::GetGraphicPackageInstanceName - can be checked, only on host, to assure the same platform implementation instance is used as the one used for generating the asset.

        c_ItemCount
    };

    /**
     * CFF asset validation levels. Asset validation levels must be used
     *  in conjunction with AssetValidationAttributes flags to define overall validation of the asset.
     */
    enum AssetValidationLevels {
        WarningLevel = 0,           ///<Issue a warning if associated attribute validation fails.
        IgnoreLevel = 1,            ///<Skip associated attribute validation.
        ErrorLevel = 2,             ///<Issue an error if associated attribute validation fails.

        c_LevelCount = 3
    };

    /**
     * Template structure that defines unique flags for each AssetValidationAttributes * AssetValdiationLevels combination.
     *  E.g. to specify that WidgetSet hash validation must issue an error AssetValidationFlag<WidgetSetHashAttribute, ErrorLevel>::Flag
     *  should be used.
     */
    template<AssetValidationAttributes attribute, AssetValidationLevels level>
    struct AssetValidationFlag {
        enum { Flag = (((static_cast<UInt>(1) << (c_LevelCount - 1)) - 1) & level) << (attribute * (c_LevelCount - 1)) };
    };

    /**
     * Interfaces for validating asset content during asset loading
     */
     class AssetValidation {
         friend class Internal::AssetSet;
        public:
            static const Int32 CURRENT_VERSION = 30600010;

            static UInt32 DefaultValidationFlags() {
                //By default warning is issued for each validation item.
                return Flag<CffVersionAttribute, ErrorLevel>() |
                    Flag<FractionalNumberRepresentationAttribute, ErrorLevel>() |
                    Flag<PlatformInstanceNameAttribute, IgnoreLevel>();
            }

            static UInt32 IgnoreAllValidationFlags() {
                return Flag<CffVersionAttribute, IgnoreLevel>() |
                    Flag<WidgetSetVersionAttribute, IgnoreLevel>() |
                    Flag<WidgetSetHashAttribute, IgnoreLevel>() |
                    Flag<CanderaVersionAttribute, IgnoreLevel>() |
                    Flag<FractionalNumberRepresentationAttribute, IgnoreLevel>() |
                    Flag<PlatformNameAttribute, IgnoreLevel>() |
                    Flag<PlatformInstanceNameAttribute, IgnoreLevel>();
            }

            template<AssetValidationAttributes attribute, AssetValidationLevels level>
            static UInt32 Flag() { return static_cast<UInt32>(AssetValidationFlag<attribute, level>::Flag); }

            static bool ValidateAssetVersion(const AssetData& assetData, UInt32 validationFlags);
            static bool ValidateAssetPlatform(const AssetData& assetData, UInt32 validationFlags);
            /**
            * Get asset version information (AssetData::AssetVersionInfo).
            *
            * Asset version information can be retrieved without a full load of the asset.
            *
            * @param assetRepository cff asset repository.
            * @param assetVersionInfo [out] asset version information.
            * @return true asset version information could be read from the given asset repository, false otherwise.
            */
            static bool GetAssetVersionInfo(AssetRepository* assetRepository, AssetData::AssetVersionInfo& assetVersionInfo);
        private:
            template<AssetValidationAttributes attribute>
            static bool ValidateAssetData(const AssetData& assetData, UInt32 validationFlags);

            template<AssetValidationAttributes attribute>
            static bool Validate(const AssetData& assetData);
    };

/** @} */ // end of AssetLoaderBase

} // namespace Candera

#endif // CANDERA_ASSET_VALIDATION_H
