//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSETREPOSITORY_H)
#define CANDERA_ASSETREPOSITORY_H

#include <Candera/Environment.h>
#include <FeatStd/Platform/CriticalSection.h>

namespace Candera {

    namespace Internal {
        class AssetSet;
    }

/** @addtogroup AssetLoaderBase
 *  @{
 */

    /**
     * @brief Abstract class providing methods to read asset (e.g. from file, from flash)
     */
    class AssetRepository
    {
        friend class Candera::Internal::AssetSet;
    public:
        /**
         *  Asset Access Mode (NonAddressable, ReadOnlyAddressable)
         */
        enum AssetAccessMode {
            NonAddressable,         ///< Asset data is only available through AssetRead.
            ReadOnlyAddressable     ///< Asset data is available through AssetRead and GetConstData.
        };

        /**
         *  Asset Seek Mark (Begin, Current)
         */
        enum AssetSeekMark {
            Begin,  ///< Begin.
            Current ///< Current.
        };

        /**
         *  Constructor
         */
        AssetRepository(void) {};

        /**
         *  Destructor
         */
        virtual ~AssetRepository(void) {};

        /**
         *  Initializes the repository so it's ready for read
         *  @return true if successful
         */
        virtual bool Initialize() = 0;

        /**
         *  Reads from the asset repository and copies output in buffer. Position indicator (AssetTell) is
         *  advanced by the total amount of bytes read.
         *  @param buffer      Pre-allocated buffer to write read data into.
         *  @param elementSize Size of a single data element, in bytes.
         *  @param count       Count of elements to read.
         *  @return amount of bytes read
         */
        virtual SizeType AssetRead(void* buffer, SizeType elementSize, SizeType count) = 0;

        /**
         *  Seeks a position in the asset by offset bytes
         *  @param offset Number of offset to look and set position indicator (AssetTell) to.
         *  @param mark   Position where offset is added to, Begin or Current Position.
         *  @return true if successful
         */
        virtual bool AssetSeek(OffsetType offset, AssetSeekMark mark) = 0;

        /**
         *  Returns the current asset repository position
         *  @return Current Position in data stream. -1 if an error occurs.
         */
        virtual OffsetType AssetTell() = 0;

        /**
         *  Returns the available asset access modes
         *  @return The available asset access modes.
         */
        virtual AssetAccessMode GetAssetAccessMode() { return NonAddressable; }

        /**
         *  Retrieves a pointer to a block within the asset reposition.
         *  The block starts from the current seek position. The position and status flags
         *  are updated the same as in AssetRead. In case of success the retrieved block
         *  has the size elementSize * count. The return value specifies the number of elements
         *  available within the block or -1 on failure.
         *  @param buffer Pointer to an object where to store the block start address.
         *  @param elementSize Size of an element within the block.
         *  @param count Number of requested elements.
         *  @return Number of elements successfully reserved. -1 on failure.
         */
        virtual SizeType GetConstData(const void ** buffer, SizeType elementSize, SizeType count) {
            FEATSTD_UNUSED(buffer);
            FEATSTD_UNUSED(elementSize);
            FEATSTD_UNUSED(count);
            return 0;
        }

        /**
         *  Finalizes the repository.
         *  @return true if successful
         */
        virtual bool Finalize() = 0;

    private:
        FeatStd::Internal::CriticalSection m_seekAndReadCriticalSection;
    };

 /** @} */ // end of AssetLoaderBase

}

#endif // CANDERA_ASSETREPOSITORY_H
