//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSETPROVIDER_H)
#define CANDERA_ASSETPROVIDER_H

#include <Candera/Environment.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Candera/EngineBase/Common/AbstractNodePointer.h>
#include <Candera/EngineBase/Common/ResourceDataHandle.h>
#include <Candera/System/GlobalizationBase/LanguagePack.h>
#include <Candera/System/Rtti/Rtti.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetId.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLibEntities.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLibHeadInformation.h>
#include <CanderaAssetLoader/AssetLoaderBase/ItemPathInfo.h>
#include <CanderaAssetLoader/AssetLoaderBase/AsyncAssetProviderProxy.h>

FEATSTD_LINT_FILE(967, AssetProvider.h, "false positive - file has valid include guards.")

namespace Candera {
/** @addtogroup AssetLoaderBase
 *  @{
 */

    // Forward declarations
    namespace Animation {
        class AnimationGroupPlayer;
        class AnimationPlayer;
        class AnimationPlayerBase;
    }
    namespace Internal {
        class AssetProviderFunctions;
        class StateMachineBehaviorData;
    }
    namespace Scripting {
        class Script;
    }

    class Bitmap;
    class Effect2D;
    class GraphicDeviceUnit;
    class Shader;
    class VertexBuffer;
    class SceneContext;
    class Node;
    class Node2D;
    class WidgetBase;
    class ItemPathInfo;
    class RenderTarget3D;
    class RenderMode;
    class Scene2DContext;
    class Theme;
    class Appearance;
    class Material;
    class RenderMode;
    class Texture;
    class AbstractShaderParamSetter;
    class BitmapImage2D;
    class BitmapTextureImage;
    class CameraGroup;
    class CompositeGroup;
    class CompositeGroup2D;
    class SharedClearMode;
    class AssetProvider;
    class SoftwareLayerProperties;
    class AssetFontStore;

    namespace TextRendering {
        class SharedStyle;
    }
    namespace Internal {
        class AnimationPropertySetterFactory;
        class AnimationPropertySetterFactory3D;
        class AnimationPropertySetterFactory2D;
        class AnimationPropertySetterFactoryCommon;
    }
    
    namespace MetaInfo {
        template <typename Type, Type(AssetProvider::*Getter)(const Candera::Internal::AssetId&)> struct AssetLoaderConverterFromString;
        namespace Internal {
            template <typename T> struct DataType;
        }
    }

    template <typename Type, typename Base, Base(AssetProvider::*Getter)(const Candera::Internal::AssetId&)> class AssetObject;

    typedef void* ResourceHandle;

    /**
     * @brief Abstract class providing methods for retrieving objects
     */
    class AssetProvider
    {
    public:
        friend class AssetFontStore;
        friend class Candera::Internal::AnimationPropertySetterFactory;
        friend class Candera::Internal::AnimationPropertySetterFactoryCommon;
        friend class Candera::Internal::AnimationPropertySetterFactory2D;
        friend class Candera::Internal::AnimationPropertySetterFactory3D;
        friend class Candera::Internal::AssetProviderFunctions;
        friend class SoftwareLayerProperties;
        template <typename T> friend struct Candera::MetaInfo::Internal::DataType;
        template <typename Type, Type(AssetProvider::*Getter)(const Candera::Internal::AssetId&)> friend struct Candera::MetaInfo::AssetLoaderConverterFromString;
        template <typename Type, typename Base, Base(AssetProvider::*Getter)(const Candera::Internal::AssetId&)> friend class Candera::AssetObject;

        /**
         * Constructor
         */
        AssetProvider(void);

        /**
         * Destructor
         */
        virtual ~AssetProvider(void);

        /**
         * Get a proxy class that allows to retrieve assets asynchronously.
         */
        AsyncAssetProviderProxy& AsyncProxy() { return m_asyncProxy; }

        /**
        * Get the ResourceDataHandle associated to the raw resource. Use ResourceObject to access ressource data.
        * @param id Id of the raw resource.
        * @return Handle to the resource in the asset.
        */
        virtual ResourceDataHandle GetRawResourceById(Id id) = 0;

        /**
        * Get the ResourceDataHandle associated to the font resource. Use ResourceObject to access font data.
        * @param id Id of the font resource.
        * @return Handle to the resource in the asset.
        */
        virtual ResourceDataHandle GetFontResourceById(Id id) = 0;

        /**
         * Returns the animation (single or group) with the specified name.
         * @param id Id of the animation to get from asset.
         * @return The animation or null if not found.
         */
        MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> GetAnimationById(Id id);

        /**
         * Release the animation with the specified id.
         * @param id Id specifying the animation to release.
         */
        void ReleaseAnimationById(Id id);

        /**
         * Returns the bitmap with the specified id.
         * @param id Id of the bitmap to get from asset.
         * @return The bitmap object or null if not found.
         */
        FeatStd::MemoryManagement::SharedPointer<Bitmap> GetBitmapById(Id id);

        /**
         * Release the bitmap with the specified id.
         * @param id Id specifying the bitmap to release.
         */
        void ReleaseBitmapById(Id id);

        /**
         * Returns the graphic device unit with the specified id.
         * @param id Id of the graphic device unit to get from asset.
         * @return The graphic device unit object or null if not found.
         */
        GraphicDeviceUnit* GetGraphicDeviceUnitById(Id id);

        /**
         * Returns the display id, width and height of the display with the specified asset id.
         * @param id The ID of the display to get from asset.
         * @param width out parameter for the display width.
         * @param height out parameter for the display height.
         * @return the id of the display or -1 if not found.
         */
        Int GetDisplayDataById(Id id, Int& width, Int& height);

        /**
         * Selects the current theme to be used for asset loading.
         * Please note that this also loads the theme from asset if it has not been loaded before.
         * It is allowed to specify a null Id. This will reset the loading to the default themes
         * (as specified in the asset for each scene).
         * @param id Id specifying theme to use for asset loading. Can be 0 to use default theme.
         */
        void SetCurrentThemeById(Id id);

        /**
         * Returns the current theme Id to be used for asset loading.
         * @return The Id of the currently selected theme.
         */
        Id GetCurrentThemeId() const;

        /**
         * Returns the style with the specified id.
         * @param id Id of the style to get from asset.
         * @return The style object or null if not found.
         */
        MemoryManagement::SharedPointer<TextRendering::SharedStyle> GetTextStyleById(Id id);

        /**
         * Returns the CameraGroup with the specified id.
         * @param id Id of the CameraGroup to get from asset.
         * @return The CameraGroup object or null if not found.
         */
        CameraGroup* GetCameraGroupById(Id id);

        /**
         * Returns the LanguagePack with the specified locale.
         * @param  locale Locale of the LanguagePack to get from asset.
         * @return The LanguagePack object or null if not found.
         */
        virtual Globalization::LanguagePack::SharedPointer GetLanguagePack(const Char* locale) = 0;

        /**
         * Returns the LanguagePack with the specified id.
         * @param  id Id of the LanguagePack to get from asset.
         * @return The LanguagePack object or null if not found.
         */
        MemoryManagement::SharedPointer<Globalization::LanguagePack> GetLanguagePackById(Id id);

        /**
         * Returns the shared ClearMode with the specified id.
         * @param id Id specifying the shared ClearMode to get.
         * @return The shared ClearMode object or null if not found.
         */
        MemoryManagement::SharedPointer<SharedClearMode> GetClearModeById(Id id);

        CANDERA_DEPRECATED_3_0_0("Please retrieve bitmaps by Id. See GetBitmapById(Id)",
            virtual FeatStd::MemoryManagement::SharedPointer<Bitmap> GetBitmap(const Char* name));

#ifdef CANDERA_2D_ENABLED
        /**
         * Returns the Scene2DContext with the specified id.
         * @param id Id specifying the Scene2DContext to get.
         * @return The Scene2DContext object or null if not found.
         */
        Scene2DContext* GetScene2DById(Id id);


        /**
         * Returns the BitmapImage2D with the specified id.
         * @param id Id specifying the BitmapImage2D to get.
         * @return The BitmapImage2D object or null if not found.
         */
        MemoryManagement::SharedPointer<BitmapImage2D> GetBitmapImage2DById(Id id);

        /**
         * Release the 2D scene with the specified id.
         * @param id Id specifying the Scene2DContext to release.
         */
        void ReleaseScene2DById(Id id);

        /**
         * @brief Create a CompositeGroup2D built based on the Composite2D with the given id.
         *
         * @param id Id of the CompositeGroup2D.
         * @return CompositeGroup2D instance.
         */
        CompositeGroup2D* CreateCompositeGroup2DById(Id id);

        /**
         * @brief Release a CompositeGroup2D built with CreateCompositeGroup2D.
         *
         * @param compositeGroup CompositeGroup2D to be released.
         */
        virtual void ReleaseCompositeGroup2D(CompositeGroup2D* compositeGroup) = 0;

        /**
        * Returns the Node2D with the specified id.
        * @param sceneId Id specifying the Scene2D that contains the node.
        * @param nodeId Id specifying the Node2D within the Scene.
        * @return The Node2D object or null if not found.
        */
        Node2D* GetNode2DById(Id sceneId, Id nodeId);

        typedef bool(*MatchingFunc2D)(const Node2D*, const Char*, const Char*);

        CANDERA_DEPRECATED_3_0_0("Please create CompositeGroup2D directly with CreateCompositeGroup2D(name)",
            virtual Scene2DContext* GetComposite2D(const Char* name));
#endif // CANDERA_2D_ENABLED

#ifdef CANDERA_3D_ENABLED
        /**
         * Returns the Shader with the specified id.
         * @param id Id specifying the Shader to get.
         * @return The Shader object or null if not found.
         */
        MemoryManagement::SharedPointer<Shader> GetShader(Id id);


        /**
         * Returns the VertexBuffer with the specified id.
         * @param id Id specifying the VertexBuffer to get.
         * @return The VertexBuffer object or null if not found.
         */
        MemoryManagement::SharedPointer<VertexBuffer> GetVertexBufferById(Id id);

        /**
         * Returns the SceneContext with the specified id.
         * @param id Id specifying the SceneContext to get.
         * @return The SceneContext object or null if not found.
         */
        SceneContext* GetSceneById(Id id);

        /**
         * Returns the referenced template with the specified id.
         * @param id Id specifying the referenced template to get.
         * @return The referenced template object or null if not found.
         */
        Node* GetReferencedTemplateById(Id id);

        /**
         * Release the scene with the specified id.
         * @param id Id specifying the scene to release.
         */
        void ReleaseSceneById(Id id);

        /**
         * Returns the Appearance with the specified id.
         * @param id Id specifying the Appearance to get.
         * @return The Appearance object or null if not found.
         */
        MemoryManagement::SharedPointer<Appearance> GetAppearanceById(Id id);

        /**
         * Returns the Material with the specified id.
         * @param id Id specifying the Material to get.
         * @return The Material object or null if not found.
         */
        MemoryManagement::SharedPointer<Material> GetMaterialById(Id id);

        /**
         * Returns the RenderMode with the specified id.
         * @param id Id specifying the RenderMode to get.
         * @return The RenderMode object or null if not found.
         */
        MemoryManagement::SharedPointer<RenderMode> GetRenderModeById(Id id);

        /**
         * Returns the AbstractShaderParamSetter with the specified id.
         * @param id Id specifying the AbstractShaderParamSetter to get.
         * @return The AbstractShaderParamSetter object or null if not found.
         */
        MemoryManagement::SharedPointer<AbstractShaderParamSetter> GetShaderParamSetterById(Id id);

        /**
         * Returns the Texture with the specified id.
         * @param id Id specifying the Texture to get.
         * @return The Texture object or null if not found.
         */
        MemoryManagement::SharedPointer<Texture> GetTextureById(Id id);

        /**
         * Returns the BitmapTextureImage with the specified id.
         * @param id Id specifying the BitmapTextureImage to get.
         * @return The BitmapTextureImage object or null if not found.
         */
        MemoryManagement::SharedPointer<BitmapTextureImage> GetBitmapTextureImageById(Id id);

        /**
         * Returns the default RenderMode read from current asset file.
         * @return The default RenderMode read from current asset file.
         */
        virtual MemoryManagement::SharedPointer<RenderMode> GetDefaultRenderMode() = 0;

        /**
         * @brief Create a CompositeGroup built based on the Composite with the given id.
         *
         * @param id Id of the Composite.
         * @return CompositeGroup instance.
         */
        CompositeGroup* CreateCompositeGroupById(Id id);

        /**
        * @brief Release a CompositeGroup built with CreateCompositeGroup.
        *
        * @param compositeGroup CompositeGroup to be released.
        */
        virtual void ReleaseCompositeGroup(CompositeGroup* compositeGroup) = 0;

        /**
        * Returns the Node with the specified id.
        * @param sceneId Id specifying the Scene that contains the node.
        * @param nodeId Id specifying the Node within the Scene.
        * @return The Node object or null if not found.
        */
        Node* GetNodeById(Id sceneId, Id nodeId);

        typedef bool(*MatchingFunc)(const Node*, const Char*, const Char*);

        CANDERA_DEPRECATED_3_0_0("Please create CompositeGroups directly with CreateCompositeGroup(name)",
            virtual SceneContext* GetComposite(const Char* name));
#endif //CANDERA_3D_ENABLED

        FEATSTD_RTTI_DECLARATION();
        private:

            /**
            * Retrieve the AssetId of an object from the asset.
            *
            * @param assetLib AssetLib that contains the object.
            * @param name Name of the object to look for.
            * @return AssetId of the object.
            */
            virtual Candera::Internal::AssetId GetAssetId(AssetLib assetLib, const Char* name);

            /**
             * Retrieve the name of an object from the asset, if available.
             *
             * @param AssetId of the object.
             * @return name of the object.A
             */
            virtual const Char* GetNameByAssetId(const Candera::Internal::AssetId& assetId) const;

            virtual ResourceHandle OpenRawResourceByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual ResourceHandle OpenFontResourceByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual Int32 GetResourceSizeInternal(ResourceHandle) { return 0; }
            virtual const void* GetResourceAddressInternal(ResourceHandle) { return 0; }
            virtual SizeType ReadResourceInternal(ResourceHandle, void*, OffsetType, SizeType) { return 0; }
            virtual void CloseResourceInternal(ResourceHandle) { }
            virtual MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> GetAnimationByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual void ReleaseAnimationByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual FeatStd::MemoryManagement::SharedPointer<Bitmap> GetBitmapByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual void ReleaseBitmapByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual GraphicDeviceUnit* GetGraphicDeviceUnitByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual Int GetDisplayDataByAssetId(const Candera::Internal::AssetId& assetId, Int& width, Int& height) = 0;
            virtual void SetCurrentThemeByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual Candera::Internal::AssetId GetCurrentThemeAssetId() const = 0;
            virtual MemoryManagement::SharedPointer<TextRendering::SharedStyle> GetTextStyleByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual CameraGroup* GetCameraGroupByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            AbstractNodePointer GetNodePointerByAssetId(const Candera::Internal::AssetId& assetId);
            virtual Candera::Internal::StateMachineBehaviorData* GetStateMachineBehaviorDataByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<Globalization::LanguagePack> GetLanguagePackByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<SharedClearMode> GetClearModeByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual WidgetBase* GetWidgetByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual WidgetBase* GetWidgetByItemPath(const ItemPathInfo& itemPathInfo) = 0;
#ifdef CANDERA_3D_ENABLED
            virtual MemoryManagement::SharedPointer<Shader> GetShaderByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<VertexBuffer> GetVertexBufferByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual Node* GetReferencedTemplateByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual SceneContext* GetSceneByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual void ReleaseSceneByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<Appearance> GetAppearanceByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<Material> GetMaterialByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<RenderMode> GetRenderModeByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<AbstractShaderParamSetter> GetShaderParamSetterByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<Texture> GetTextureByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<BitmapTextureImage> GetBitmapTextureImageByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual CompositeGroup* CreateCompositeGroupByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual Node* GetNodeByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            static AssetObjectType GetNodeTypePrivate(const Node* node);
            static bool IsMatchPrivate(const Node* node, const Char* name, const Char* type);
            static Node* SearchNodeByPathPrivate(const ItemPathInfo& nodePathInfo, const Node* rootNode, MatchingFunc matchingFunction = 0);
            virtual Node* GetRootNode(AssetObjectType aoType, const Char* name) = 0;
            virtual bool GetCompositePropertyInfo(CompositeGroup* compositeGroup, const Char* propertyName, WidgetBase*& widget, const Char*& widgetPropertyName) = 0;
#endif //CANDERA_3D_ENABLED
#ifdef CANDERA_2D_ENABLED
            virtual Scene2DContext* GetScene2DByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<BitmapImage2D> GetBitmapImage2DByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual void ReleaseScene2DByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual CompositeGroup2D* CreateCompositeGroup2DByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual Node2D* GetNode2DByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            virtual MemoryManagement::SharedPointer<Effect2D> GetEffect2DByAssetId(const Candera::Internal::AssetId& assetId) = 0;
            static bool IsMatch2DPrivate(const Node2D* node, const Char* name, const Char* type);
            static AssetObjectType GetNode2DTypePrivate(const Node2D* node);
            static Node2D* SearchNode2DByPathPrivate(const ItemPathInfo& nodePathInfo, const Node2D* rootNode, MatchingFunc2D matchingFunction = 0);
            virtual Node2D* GetRootNode2D(AssetObjectType aoType, const Char* name) = 0;
            virtual bool GetComposite2DPropertyInfo(CompositeGroup2D* compositeGroup, const Char* propertyName, WidgetBase*& widget, const Char*& widgetPropertyName) = 0;
#endif //CANDERA_2D_ENABLED
#ifdef CANDERA_SCRIPTING_ENABLED
            virtual FeatStd::MemoryManagement::SharedPointer<Scripting::Script> GetScriptByAssetId(const Candera::Internal::AssetId& assetId);
#endif //CANDERA_SCRIPTING_ENABLED
            MemoryManagement::SharedPointer<Appearance> GetAppearanceFromNode(const ItemPathInfo& appearancePath);

            AssetProvider& GetThis() { return *this; }

            AsyncAssetProviderProxy m_asyncProxy;
    };
  
 /** @} */ // end of AssetLoaderBase
}
#endif // CANDERA_ASSETPROVIDER_H
