//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AssetProvider.h"

#ifdef CANDERA_3D_ENABLED
#include <CanderaAssetLoader/AssetLoader3D/SceneContext.h>
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/Core/StereoCamera.h>
#include <Candera/Engine3D/Core/Light.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/PlanarShadow.h>
#include <Candera/Engine3D/Core/Billboard.h>
#include <Candera/Engine3D/Core/ReflectionCamera.h>
#include <Candera/Engine3D/Core/PointSprite.h>
#include <Candera/Engine3D/Core/LodNode.h>
#include <Candera/Engine3D/Core/MorphingMesh.h>
#include <Candera/Engine3D/Core/CompositeGroup.h>
#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
#include <CanderaAssetLoader/AssetLoader2D/Scene2DContext.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Core/Group2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/CompositeGroup2D.h>
#include <Candera/Engine2D/Core/BitmapImage2D.h>
#endif //CANDERA_2D_ENABLED


#ifdef CANDERA_SCRIPTING_ENABLED
#include <CanderaScripting/Script.h>
#endif //CANDERA_SCRIPTING_ENABLED

#include <Candera/EngineBase/Animation/AnimationGroupPlayer.h>
#include <Candera/EngineBase/Animation/AnimationPlayer.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <Candera/TextEngine/Style.h>

#include <CanderaPlatform/OS/StringPlatform.h>
#include <CanderaAssetLoader/AssetLoaderBase/Theme.h>

#include <Candera/System/Diagnostics/Log.h>

namespace Candera {

    using namespace Diagnostics;
    using namespace Internal;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);
    FEATSTD_RTTI_BASECLASS_DEFINITION(AssetProvider)

    AssetProvider::AssetProvider(void) :
    m_asyncProxy(GetThis())
    {
    }

    AssetProvider::~AssetProvider(void)
    {
    }

    MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> AssetProvider::GetAnimationById(Id id)
    {
        MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> result = GetAnimationByAssetId(AssetIdFunctions::CreateId(AnimationLib, AotUnknown, id, 1));
        if (result == 0) {
            result = GetAnimationByAssetId(AssetIdFunctions::CreateId(AnimationGroupLib, AotUnknown, id, 1));
        }
        return result;
    }

    void AssetProvider::ReleaseAnimationById(Id id)
    {
        //id is unique (SC restriction).
        ReleaseAnimationByAssetId(AssetIdFunctions::CreateId(AnimationLib, AotUnknown, id, 1));
        ReleaseAnimationByAssetId(AssetIdFunctions::CreateId(AnimationGroupLib, AotUnknown, id, 1));
    }

    FeatStd::MemoryManagement::SharedPointer<Bitmap> AssetProvider::GetBitmapById(Id id)
    {
        return GetBitmapByAssetId(AssetIdFunctions::CreateId(BitmapLib, AotUnknown, id, 1));
    }

    void AssetProvider::ReleaseBitmapById(Id id)
    {
        ReleaseBitmapByAssetId(AssetIdFunctions::CreateId(BitmapLib, AotUnknown, id, 1));
    }

    GraphicDeviceUnit* AssetProvider::GetGraphicDeviceUnitById(Id id)
    {
        return GetGraphicDeviceUnitByAssetId(AssetIdFunctions::CreateId(RenderTargetLib, AotUnknown, id, 1));
    }

    Int AssetProvider::GetDisplayDataById(Id id, Int& width, Int& height)
    {
        return GetDisplayDataByAssetId(AssetIdFunctions::CreateId(DisplayLib, AotUnknown, id, 1), width, height);
    }

    void AssetProvider::SetCurrentThemeById(Id id)
    {
        SetCurrentThemeByAssetId(AssetIdFunctions::CreateId(ThemeLib, AotUnknown, id, 1));
    }

    Id AssetProvider::GetCurrentThemeId() const
    {
        return AssetIdFunctions::GetLibraryId(GetCurrentThemeAssetId());
    }

    MemoryManagement::SharedPointer<TextRendering::SharedStyle> AssetProvider::GetTextStyleById(Id id)
    {
        return GetTextStyleByAssetId(AssetIdFunctions::CreateId(TextStyleLib, AotUnknown, id, 1));
    }

    CameraGroup* AssetProvider::GetCameraGroupById(Id id)
    {
        return GetCameraGroupByAssetId(AssetIdFunctions::CreateId(CameraGroupLib, AotUnknown, id, 1));
    }

    AbstractNodePointer AssetProvider::GetNodePointerByAssetId(const Candera::Internal::AssetId& assetId)
    {
#ifdef CANDERA_3D_ENABLED
        if ((assetId.m_libraryType == Candera::CompositeLib) || (assetId.m_libraryType == Candera::SceneLib)) {
            return AbstractNodePointer(GetNodeByAssetId(assetId));
        }
#endif
#ifdef CANDERA_2D_ENABLED
        if ((assetId.m_libraryType == Candera::Composite2DLib) || (assetId.m_libraryType == Candera::Scene2DLib)) {
            return AbstractNodePointer(GetNode2DByAssetId(assetId));
        }
#endif
        return AbstractNodePointer();
    }

    MemoryManagement::SharedPointer<Globalization::LanguagePack> AssetProvider::GetLanguagePackById(Id id)
    {
        return GetLanguagePackByAssetId(AssetIdFunctions::CreateId(LanguagePackLib, AotUnknown, id, 1));
    }

    MemoryManagement::SharedPointer<SharedClearMode> AssetProvider::GetClearModeById(Id id)
    {
        return GetClearModeByAssetId(AssetIdFunctions::CreateId(ClearModeLib, AotUnknown, id, 1));
    }

#ifdef CANDERA_3D_ENABLED

    MemoryManagement::SharedPointer<Shader> AssetProvider::GetShader(Id id)
    {
        return GetShaderByAssetId(AssetIdFunctions::CreateId(ShaderLib, AotUnknown, id, 1));
    }

    MemoryManagement::SharedPointer<VertexBuffer> AssetProvider::GetVertexBufferById(Id id)
    {
        return GetVertexBufferByAssetId(AssetIdFunctions::CreateId(VertexBufferLib, AotUnknown, id, 1));
    }

    SceneContext* AssetProvider::GetSceneById(Id id)
    {
        return GetSceneByAssetId(AssetIdFunctions::CreateId(SceneLib, AotScene, id, 1));
    }

    Node* AssetProvider::GetReferencedTemplateById(Id id)
    {
        return GetReferencedTemplateByAssetId(AssetIdFunctions::CreateId(ReferencedTemplateLib, AotUnknown, id, 1));
    }

    void AssetProvider::ReleaseSceneById(Id id)
    {
        ReleaseSceneByAssetId(AssetIdFunctions::CreateId(SceneLib, AotUnknown, id, 1));
    }

    MemoryManagement::SharedPointer<Appearance> AssetProvider::GetAppearanceById(Id id)
    {
        return GetAppearanceByAssetId(AssetIdFunctions::CreateId(AppearanceLib, AotUnknown, id, 1));
    }

    MemoryManagement::SharedPointer<Material> AssetProvider::GetMaterialById(Id id)
    {
        return GetMaterialByAssetId(AssetIdFunctions::CreateId(MaterialLib, AotUnknown, id, 1));
    }

    MemoryManagement::SharedPointer<RenderMode> AssetProvider::GetRenderModeById(Id id)
    {
        return GetRenderModeByAssetId(AssetIdFunctions::CreateId(RenderModeLib, AotUnknown, id, 1));
    }

    MemoryManagement::SharedPointer<AbstractShaderParamSetter> AssetProvider::GetShaderParamSetterById(Id id)
    {
        return GetShaderParamSetterByAssetId(AssetIdFunctions::CreateId(ShaderParamSetterLib, AotUnknown, id, 1));
    }

    MemoryManagement::SharedPointer<Texture> AssetProvider::GetTextureById(Id id)
    {
        return GetTextureByAssetId(AssetIdFunctions::CreateId(TextureLib, AotUnknown, id, 1));
    }

    MemoryManagement::SharedPointer<BitmapTextureImage> AssetProvider::GetBitmapTextureImageById(Id id)
    {
        return GetBitmapTextureImageByAssetId(AssetIdFunctions::CreateId(BitmapLib, AotUnknown, id, 1));
    }

    CompositeGroup* AssetProvider::CreateCompositeGroupById(Id id)
    {
        return CreateCompositeGroupByAssetId(AssetIdFunctions::CreateId(CompositeLib, AotUnknown, id, 1));
    }

#endif //CANDERA_3D_ENABLED

#ifdef CANDERA_2D_ENABLED
    Scene2DContext* AssetProvider::GetScene2DById(Id id)
    {
        return GetScene2DByAssetId(AssetIdFunctions::CreateId(Scene2DLib, AotScene2D, id, 1));
    }

    MemoryManagement::SharedPointer<BitmapImage2D> AssetProvider::GetBitmapImage2DById(Id id)
    {
        return GetBitmapImage2DByAssetId(AssetIdFunctions::CreateId(BitmapLib, AotUnknown, id, 1));
    }

    void AssetProvider::ReleaseScene2DById(Id id)
    {
        ReleaseScene2DByAssetId(AssetIdFunctions::CreateId(Scene2DLib, AotUnknown, id, 1));
    }

    CompositeGroup2D* AssetProvider::CreateCompositeGroup2DById(Id id)
    {
        return CreateCompositeGroup2DByAssetId(AssetIdFunctions::CreateId(Composite2DLib, AotUnknown, id, 1));
    }

    SceneContext* AssetProvider::GetComposite(const Char* /*name*/)
    {
        return 0;
    }
#endif //CANDERA_2D_ENABLED

#ifdef CANDERA_3D_ENABLED

    MemoryManagement::SharedPointer<Texture> GetTextureFromAppearance(MemoryManagement::SharedPointer<Appearance> appearance, const Char* textureName)
    {
        MemoryManagement::SharedPointer<Texture> texture(0);
        for (UInt i = 0; i < CANDERA_MAX_TEXTURE_UNIT_COUNT; i++) {
            MemoryManagement::SharedPointer<Texture> currentTexture = appearance->GetTexture(i);

            if (currentTexture.PointsToNull() || (currentTexture->GetName() == 0)) {
                continue;
            }

            if (StringPlatform::CompareStrings(textureName, currentTexture->GetName()) == 0) {
                texture = currentTexture;
            }
        }
        return texture;
    }

    MemoryManagement::SharedPointer<Appearance> GetAppearanceFromAppearanceCollection(MemoryManagement::SharedPointer<Appearance> firstAppearance, const Char* appearanceName)
    {
        while (!firstAppearance.PointsToNull()) {
            if ((firstAppearance->GetName() != 0) && (StringPlatform::CompareStrings(appearanceName, firstAppearance->GetName()) == 0)) {
                return firstAppearance;
            }
            else {
                firstAppearance = firstAppearance->GetNextPass();
            }
        }

        return MemoryManagement::SharedPointer<Appearance>(0);
    }

    MemoryManagement::SharedPointer<Appearance> AssetProvider::GetAppearanceFromNode(const ItemPathInfo& appearancePath)
    {
        const Char* appearanceName = appearancePath.GetName(appearancePath.GetDepth() - 1);
        if (appearanceName == 0) {
            return MemoryManagement::SharedPointer<Appearance>(0);
        }

        // Node/AppearanceCollection/Appearance
        ItemPathInfo assocNodePathInfo(appearancePath, appearancePath.GetDepth() - 2);

        Node* assocNode = AssetProvider::SearchNodeByPathPrivate(assocNodePathInfo, GetRootNode(assocNodePathInfo.GetType(0), assocNodePathInfo.GetName(0)));
        if (assocNode == 0) {
            return MemoryManagement::SharedPointer<Appearance>(0);
        }

        return GetAppearanceFromAppearanceCollection(assocNode->GetAppearance(), appearanceName);
    }

    Scene2DContext* AssetProvider::GetComposite2D(const Char* /*name*/)
    {
        return 0;
    }
#endif


#ifdef CANDERA_3D_ENABLED



    Node* AssetProvider::SearchNodeByPathPrivate(const ItemPathInfo& nodePathInfo, const Node* rootNode, MatchingFunc matchingFunction /* = 0 */)
    {
        if (rootNode == 0) {
            return 0;
        }

        if (!nodePathInfo.IsPathValid()) {
            return 0;
        }

        if (matchingFunction == 0) {
            matchingFunction = &AssetProvider::IsMatchPrivate;
        }

        const Node* searchedNode = rootNode;
        Node* foundNode = 0;
        UInt16 currentNameIndex = 1;

        if ((*matchingFunction)(rootNode, nodePathInfo.GetName(0), nodePathInfo.GetTypeStr(0))) {
            const Node* currentNode = searchedNode;

            while ((currentNode != 0) && (currentNode->GetName() != 0)) {
                if (currentNameIndex > (nodePathInfo.GetDepth() - 1)) {
                    foundNode = const_cast<Node*>(currentNode);
                    break;
                } else {
                    bool requiredNameMatched = false;

                    const Char* nameToMatch = nodePathInfo.GetName(currentNameIndex);
                    const Char* typeToMatch = nodePathInfo.GetTypeStr(currentNameIndex);
                    if (nodePathInfo.GetType(currentNameIndex) == AotAnchor) {
                        const CompositeGroup* compositeGroup = Dynamic_Cast<const CompositeGroup*>(currentNode);
                        if (compositeGroup != 0) {
                            currentNode = compositeGroup->GetAnchorPointNodeByName(nameToMatch);
                            requiredNameMatched = (currentNode != 0);
                        }
                    }
                    else if (currentNode->IsTypeOf(Camera::GetTypeId()) && (currentNameIndex < nodePathInfo.GetDepth()) && (nodePathInfo.GetType(currentNameIndex) == AotSkyBoxMesh)) {
                        const Camera* camera = Dynamic_Cast<const Camera*>(currentNode);
                        if (camera != 0) {
                            Mesh* skyBox = (camera->GetClearMode().GetSkyBox() == 0) ? 0 : camera->GetClearMode().GetSkyBox()->GetSkyBoxMesh();
                            if ((skyBox != 0) && (skyBox->GetName() != 0) && (StringPlatform::CompareStrings(skyBox->GetName(), nodePathInfo.GetName(currentNameIndex)) == 0)) {
                                currentNode = skyBox;
                                requiredNameMatched = true;
                            }
                        }
                    }
                    else {
                        const Node* child = currentNode->GetFirstChild();
                        while (child != 0) {
                            if ((*matchingFunction)(child, nameToMatch, typeToMatch)) {
                                currentNode = child;
                                requiredNameMatched = true;
                                break;
                            }
                            child = child->GetNextSibling();
                        }
                    }

                    if (!requiredNameMatched) {
                        foundNode = 0;
                        break;
                    }
                }
                currentNameIndex++;
            }
        }

        return foundNode;
    }

    AssetObjectType AssetProvider::GetNodeTypePrivate(const Node* node)
    {
        if (node == 0) {
            return AotUnknown;
        }

        // TO DO: change implementation
        // Candera RTTI provides GetDynamicTypeId which can be used to check
        // instance type id

        AssetObjectType result = AotUnknown;

        if (node->IsTypeOf(Group::GetTypeId())) {
            if (node->IsTypeOf(Scene::GetTypeId())) {
                result = AotScene;
            } else if (node->IsTypeOf(CompositeGroup::GetTypeId())) {
                result = AotCompositeNode;
            } else {
                result = AotGroup;
            }
        } else if (node->IsTypeOf(Mesh::GetTypeId())) {
            if (node->IsTypeOf(MorphingMesh::GetTypeId())) {
                result = AotMorphingMesh;
            } else if (node->IsTypeOf(PlanarShadow::GetTypeId())) {
                result = AotPlanarShadow;
            } else {
                result = AotMesh;
            }
        } else if (node->IsTypeOf(Camera::GetTypeId())) {
            if (node->IsTypeOf(ReflectionCamera::GetTypeId())) {
                result = AotReflectionCamera;
            } else {
                result = AotCamera;
            }
        } else if (node->IsTypeOf(Light::GetTypeId())) {
            result = AotLight;
        } else if (node->IsTypeOf(Billboard::GetTypeId())) {
            result = AotBillboard;
        } else if (node->IsTypeOf(PointSprite::GetTypeId())) {
            result = AotPointSprite;
        } else if (node->IsTypeOf(LodNode::GetTypeId())) {
            result = AotLodNode;
        } else if (node->IsTypeOf(StereoCamera::GetTypeId())) {
            result = AotStereoCamera;
        } else {
            //do nothing
        }


        return result;
    }

    bool AssetProvider::IsMatchPrivate(const Node* node, const Char* name, const Char* type)
    {
        if ((name == 0) || (type == 0) || (node->GetName() == 0)) {
            return false;
        }

        AssetObjectType nodeType = GetNodeTypePrivate(node);
        AssetObjectType matchType = ItemPathInfo::ConvertToAssetObjectType(type);

        bool result = (node->GetName() != 0) &&
            (StringPlatform::CompareStrings(node->GetName(), name) == 0) &&
            ((nodeType == matchType) || ((nodeType == AotCompositeNode) && (matchType == AotComposite)));

        return result;
    }


    Node* AssetProvider::GetNodeById(Id sceneId, Id nodeId)
    {
        return GetNodeByAssetId(AssetIdFunctions::CreateId(SceneLib, AotUnknown, sceneId, nodeId));
    }
#endif //CANDERA_3D_ENABLED


#ifdef CANDERA_2D_ENABLED

    Node2D* AssetProvider::SearchNode2DByPathPrivate(const ItemPathInfo& nodePathInfo, const Node2D* rootNode, MatchingFunc2D matchingFunction)
    {
        if (rootNode == 0) {
            return 0;
        }

        if (!nodePathInfo.IsPathValid()) {
            return 0;
        }

        if (matchingFunction == 0) {
            matchingFunction = &AssetProvider::IsMatch2DPrivate;
        }

        const Node2D* searchedNode = rootNode;
        Node2D* foundNode = 0;
        UInt16 currentNameIndex = 1;

        if ((*matchingFunction)(rootNode, nodePathInfo.GetName(0), nodePathInfo.GetTypeStr(0))) {
            const Node2D* currentNode = searchedNode;

            while ((currentNode != 0) && (currentNode->GetName() != 0)) {
                if (currentNameIndex > (nodePathInfo.GetDepth() - 1)) {
                    foundNode = const_cast<Node2D*>(currentNode);
                    break;
                } else {
                    bool requiredNameMatched = false;

                    const Char* nameToMatch = nodePathInfo.GetName(currentNameIndex);
                    const Char* typeToMatch = nodePathInfo.GetTypeStr(currentNameIndex);

                    if (nodePathInfo.GetType(currentNameIndex) == AotAnchor2D) {
                        const CompositeGroup2D* compositeGroup = Dynamic_Cast<const CompositeGroup2D*>(currentNode);
                        if (compositeGroup != 0) {
                            currentNode = compositeGroup->GetAnchorPointNodeByName(nameToMatch);
                            requiredNameMatched = (currentNode != 0);
                        }
                    } else {
                        const Node2D* child = currentNode->GetFirstChild();
                        while (child != 0) {
                            if ((*matchingFunction)(child, nameToMatch, typeToMatch)) {
                                currentNode = child;
                                requiredNameMatched = true;
                                break;
                            }
                            child = child->GetNextSibling();
                        }
                    }

                    if (!requiredNameMatched) {
                        foundNode = 0;
                        break;
                    }
                }
                currentNameIndex++;
            }
        }

        return foundNode;
    }


    AssetObjectType AssetProvider::GetNode2DTypePrivate(const Node2D* node)
    {
        if (node == 0) {
            return AotUnknown;
        }

        AssetObjectType result = AotUnknown;

        if (node->IsTypeOf(Group2D::GetTypeId())) {
            if (node->IsTypeOf(Scene2D::GetTypeId())) {
                result = AotScene2D;
            } else if (node->IsTypeOf(CompositeGroup2D::GetTypeId())) {
                result = AotCompositeNode2D;
            } else {
                result = AotGroup2D;
            }
        }  else if (node->IsTypeOf(RenderNode::GetTypeId())) {
            result = AotRenderNode;
        } else if (node->IsTypeOf(Camera2D::GetTypeId())) {
            result = AotCamera2D;
        } else {
            //do nothing
        }

        return result;
    }

    bool AssetProvider::IsMatch2DPrivate(const Node2D* node, const Char* name, const Char* type)
    {
        if ((name == 0) || (type == 0) || (node->GetName() == 0)) {
            return false;
        }

        AssetObjectType nodeType = GetNode2DTypePrivate(node);
        AssetObjectType matchType = ItemPathInfo::ConvertToAssetObjectType(type);

        bool result = (node->GetName() != 0) &&
            (StringPlatform::CompareStrings(node->GetName(), name) == 0) &&
            ((nodeType == matchType) || ((nodeType == AotCompositeNode2D) && (matchType == AotComposite2D)));

        return result;
    }

    Node2D* AssetProvider::GetNode2DById(Id sceneId, Id nodeId)
    {
        return GetNode2DByAssetId(AssetIdFunctions::CreateId(Scene2DLib, AotUnknown, sceneId, nodeId));
    }
#endif // CANDERA_2D_ENABLED


    AssetId AssetProvider::GetAssetId(AssetLib /*assetLib*/, const Char* /*name*/)
    {
        return AssetId::InvalidId();
    }

    const Char* AssetProvider::GetNameByAssetId(const AssetId& /*id*/) const
    {
        return 0;
    }

    Bitmap::SharedPointer AssetProvider::GetBitmap(const Char* name)
    {
        return GetBitmapByAssetId(GetAssetId(BitmapLib, name));
    }

#ifdef CANDERA_SCRIPTING_ENABLED
    FeatStd::MemoryManagement::SharedPointer<Scripting::Script> AssetProvider::GetScriptByAssetId(const Candera::Internal::AssetId& assetId)
    {
        FEATSTD_UNUSED(assetId);
        return FeatStd::MemoryManagement::SharedPointer<Scripting::Script>(0);
    }
#endif //CANDERA_SCRIPTING_ENABLED
    
}
