//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSETLOADERDATATYPETEMPLATE_H)
#define CANDERA_ASSETLOADERDATATYPETEMPLATE_H

#include <Candera/EngineBase/DataTypes/CanderaObjectDataType.h>
#include <CanderaAssetLoader/AssetLoaderBase/ContentLoader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <Candera/EngineBase/Common/AbstractNodePointer.h>

namespace Candera {
    namespace MetaInfo {

        /**
         * @brief Converter from string for asset object references.
         *
         * @tparam T Type of object to be converted.
         * @tparam ObjectType AssetObjectType associated to the type of object to be converted.
         */
        template <typename Type, Type(AssetProvider::*Getter)(const Candera::Internal::AssetId&)>
        struct AssetLoaderConverterFromString{
            /**
             * Retrieve the pointer to the asset object referenced by the item path.
             *
             * @param [out] object Pointer to the converted object.
             * @param       buf Item path.
             * @param       assetProvider Provider used to retrieve the asset object.
             * @return true if the item path could be resolved and the referenced item has the same
             *      type as the one specified by the ObjectType template parameter.
             */
            static bool ConvertFromString(Type &object, const Char *buf, AssetProvider* assetProvider = 0);
        };
        
        namespace Internal
        {
            namespace Util {
                template <typename Type>
                inline Type NullPointer()
                {
                    return Type();
                }

                template <typename Type>
                inline bool IsPointerValid(const Type& ptr)
                {
                    return (ptr != 0);
                }

                template <>
                inline AbstractNodePointer NullPointer()
                {
                    return AbstractNodePointer();
                }

                template <>
                inline bool IsPointerValid(const AbstractNodePointer& ptr)
                {
                    return ptr.IsValid();
                }
            }
        }

        template <typename Type, Type(AssetProvider::*Getter)(const Candera::Internal::AssetId&)>
        inline bool AssetLoaderConverterFromString<Type, Getter>::ConvertFromString(Type &object, const Char *buf, AssetProvider* assetProvider)
        {
            FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);
            object = Internal::Util::NullPointer<Type>();

            UInt32 tab[4] = { 0 };
            if (SerializationPlatform::ToBinary<UInt32*>(buf, ",", &tab[0], &tab[4]) != 4) {
                FEATSTD_LOG_ERROR("Failed to deserialize AssetId");
                return false;
            }
            Candera::Internal::AssetId objectId = Candera::Internal::AssetIdFunctions::GetAssetId(tab);
            if (!objectId.IsValid()) {
                FEATSTD_LOG_INFO("No object to convert.");
                return false;
            }

            if (assetProvider == 0) {
                assetProvider = ContentLoader::GetInstance().GetAssetProvider();
                if (assetProvider == 0) {
                    FEATSTD_LOG_ERROR("No associated AssetProvider to obtain Image " AssetIdLogStr, AssetIdLogArgs(objectId));
                    return false;
                }
            }

            DefaultAssetProvider* defaultAssetProvider = Dynamic_Cast<DefaultAssetProvider*>(assetProvider);
            bool isIterativeLoadingEnabled = false;
            if (defaultAssetProvider != 0) {
                isIterativeLoadingEnabled = defaultAssetProvider->IsIterativeLoadingEnabled();
                defaultAssetProvider->SetIterativeLoadingEnabled(false);
            }
            object = (assetProvider->*Getter)(objectId);
            if (defaultAssetProvider != 0) {
                defaultAssetProvider->SetIterativeLoadingEnabled(isIterativeLoadingEnabled);
            }

            if (!Internal::Util::IsPointerValid(object)) {
                FEATSTD_LOG_ERROR("Object " AssetIdLogStr " not found!", AssetIdLogArgs(objectId));
                return false;
            }

            return true;
        }


    }
}
#endif // CANDERA_ASSETLOADERDATATYPETEMPLATE_H
