//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSETLIBENTITIES_H)
#define CANDERA_ASSETLIBENTITIES_H

#include <Candera/Environment.h>

/** @addtogroup AssetLoaderBase
*  @{
*/

/**
* @brief The asset object type enumeration.
*/
enum AssetObjectType
{
    AotUnknown = 0,         ///< Unknown asset object type.
    ValuesCount = 1 << 14,

    //Category: 3 bits
    __CategoryBitPos = 11,
    _Scene     = 1 << __CategoryBitPos, ///< Scene main category flag.
    _Animation = 2 << __CategoryBitPos, ///< Animation main category flag.
    _Theme     = 3 << __CategoryBitPos, ///< Theme main category flag.
    _Misc      = 4 << __CategoryBitPos, ///< Miscellaneous main category flag.

    //Subcategory: 3 bits
    __SubcategoryBitPos = 8,
    _Subcategory1 = 1 << __SubcategoryBitPos,   ///< 1st subcategory flag.
    _Subcategory2 = 2 << __SubcategoryBitPos,   ///< 2nd subcategory flag.
    _Subcategory3 = 3 << __SubcategoryBitPos,   ///< 3rd subcategory flag.
    _Subcategory4 = 4 << __SubcategoryBitPos,   ///< 4th subcategory flag.
    _Subcategory5 = 5 << __SubcategoryBitPos,   ///< 5th subcategory flag.
    _Subcategory6 = 6 << __SubcategoryBitPos,   ///< 6th subcategory flag.
    _Subcategory7 = 7 << __SubcategoryBitPos,   ///< 7th subcategory flag.

    //Engine: 2 bits
    _Candera2D = 1 << 7,    ///< Candera2D engine category flag.
    _Candera3D = 1 << 6,    ///< Candera3D engine category flag.

    //Collection flag: 1 bit
    _Collection = 1 << 5,   ///< Collection flag.

    //Nodes
    _SceneNode = _Scene | _Subcategory1,    ///< "Scene/Scene node" subcategory flag.

    ///@{
    //Attachments
    _SceneAttachment = _Scene | _Subcategory2,  ///< "Scene/Scene attachment" subcategory flag.
    AotWidget,                                  ///< Scene attachment - Widget.
    AotWidgetGroup,                             ///< Scene attachment - WidgetGroup.
    AotBehavior,                                ///< Scene attachment - Behavior.
    AotBehaviorGroup,                           ///< Scene attachment - BehaviorGroup.
    AotAnchorDefinition,                        ///< Scene attachment - AnchorDefinition.
    AotCompositePropertyDefinition,             ///< Scene attachment - CompositePropertyDefinition.
    AotCompositeProperties,                     ///< Scene attachment - CompositeProperties.
    AotBehaviorBlock,                           ///< Scene attachment - BehaviorBlock.
    ///}@

    //Nodes
    _SceneNode2D = _SceneNode | _Candera2D, ///< "Scene/Scene node/Candera2D" subcategory flag.
    AotScene2D,                             ///< 2D Scene node - Scene2D.
    AotComposite2D,                         ///< 2D Scene node - Composite.

    AotAnchor2D,                            ///< 2D Scene node - AnchorNode.
    AotCamera2D,                            ///< 2D Scene node - CameraNode.
    AotCompositeNode2D,                     ///< 2D Scene node - CompositeNode.
    AotGroup2D,                             ///< 2D Scene node - GroupNode.
    AotRenderNode,                          ///< 2D Scene node - RenderNode.
    AotTextNode2D,                          ///< 2D Scene node - TextNode2D.

    //Attachments
    _SceneAttachment2D = _SceneAttachment | _Candera2D, ///< "Scene/Scene attachment/Candera2D" subcategory flag.
    AotEffect2D,                                        ///< 2D Scene attachment - Effect
    AotLayoutProperties,                                ///< 2D Scene attachment - LayoutProperties
    AotStackLayout,                                     ///< 2D Scene attachment - StackLayout
    AotGridLayout,                                      ///< 2D Scene attachment - GridLayout

    //Nodes
    _SceneNode3D = _SceneNode | _Candera3D,     ///< "Scene/Scene node/Candera3D" subcategory flag.
    AotScene,                                   ///< 3D Scene node - Scene node.
    AotComposite,                               ///< 3D Scene node - Composite.
    AotAnchor,                                  ///< 3D Scene node - Anchor node.
    AotBillboard,                               ///< 3D Scene node - Billboard node.
    AotCamera,                                  ///< 3D Scene node - Camera node.
    AotCompositeNode,                           ///< 3D Scene node - Composite node.
    AotGroup,                                   ///< 3D Scene node - Group node.
    AotLight,                                   ///< 3D Scene node - Light node.
    AotLodNode,                                 ///< 3D Scene node - LodNode.
    AotMesh,                                    ///< 3D Scene node - MeshNode.
    AotMorphingMesh,                            ///< 3D Scene node - MorphingMesh node.
    AotPlanarShadow,                            ///< 3D Scene node - PlanarShadow node.
    AotPointSprite,                             ///< 3D Scene node - PointSprite node.
    AotReflectionCamera,                        ///< 3D Scene node - ReflectionCamera node.
    AotSkyBoxMesh,                              ///< 3D Scene node - SkyBox node.
    AotStereoCamera,                            ///< 3D Scene node - StereoCamera node.    
    AotCanvas,                                  ///< 3D Scene node - Canvas node.
    AotCanvasGroup,                             ///< 3D Scene node - CanvasGroup node.
    AotCanvasRenderable,                        ///< 3D Scene node - CanvasRenderable node.
    AotCanvasSprite,                            ///< 3D Scene node - CanvasSprite node
    AotCanvasText,                              ///< 3D Scene node - CanvasText node.
     
    //Attachments
    _SceneAttachment3D = _SceneAttachment | _Candera3D, ///< "Scene/Scene attachment/Candera3D" subcategory flag.
    AotAppearance,      ///< 3D Scene attachment - Appearance.
    AotMaterial,        ///< 3D Scene attachment - Material.
    AotRenderMode,      ///< 3D Scene attachment - RenderMode.
    AotUniformSetter,   ///< 3D Scene attachment - UniformSetter.
    AotShaderProgram,   ///< 3D Scene attachment - ShaderProgram.
    AotTexture,         ///< 3D Scene attachment - Texture.
    AotCubeMapTexture,  ///< 3D Scene attachment - CubeMapTexture.

    AotRenderOrderBin,  ///< 3D Scene attachment - RenderOrderBin.
    AotMorphItem,       ///< 3D Scene attachment - MorphItem.

    AotDiscreteLodStrategy,             ///< 3D Scene attachment - DiscreteLodStrategy.
    AotBlendLodStrategy,                ///< 3D Scene attachment - BlendLodStrategy.
    AotGenericLodCriterion,             ///< 3D Scene attachment - GenericLodCriterion.
    AotDistanceToCameraLodCriterion,    ///< 3D Scene attachment - DistanceToCameraLodCriterion.
    AotClearMode,                       ///< 3D Scene attachment - ClearMode.

    AotCameraGroup,                     ///< 3D Scene attachment - CameraGroup.

    AotAnimation = _Animation | _Subcategory1,  ///< "Animation/Animation" subcategory flag.
    AotAnimationProperty,                       ///< Animation - AnimationProperty.
    AotAnimationKeyframeSequence,               ///< Animation - AnimationKeyframeSequence.
    AotAnimableProperty,                        ///< Animation - AnimableProperty (Animateable property).
    AotCompositeAnimation,                      ///< Animation - CompositeAnimation.
    AotCompositeNodeAnimation,                  ///< Animation - CompositeNodeAnimation.
    AotAnimationGroup,                          ///< Animation - AnimationGroup.
    AotAnimationNode,                           ///< Animation - AnimationNode.
    AotCompositeNodeAnimationNode,

    //Interpolations
    _AnimationInterpolation = _Animation | _Subcategory2,   ///< "Animation/Animation interpolation functions" subcategory flag.
    AotBezierInterpolation,                                 ///< Animation interpolation - Bezier function.
    AotEaseInterpolation,                                   ///< Animation interpolation - Ease function.
    AotLinearInterpolation,                                 ///< Animation interpolation - Linear.
    AotSplineInterpolation,                                 ///< Animation interpolation - Spline.
    AotStepInterpolation,                                   ///< Animation interpolation - Step.

    //Ease Functions
    _AnimationEaseFunction = _Animation | _Subcategory3,    ///< "Animation/Animation ease function" subcategory flag.
    AotEaseBackFunction,                                    ///< Animation ease function - Ease back.
    AotEaseBounceFunction,                                  ///< Animation ease function - Bounce.
    AotEaseElasticFunction,                                 ///< Animation ease function - Elastic.
    AotEaseExponentialFunction,                             ///< Animation ease function - Exponential.
    AotEasePowerFunction,                                   ///< Animation ease function - Power.

    AotTheme = _Theme | _Subcategory1,  ///< "Theme/Theme" subcategory flag.
    AotThemeEntry,                      ///< Theme - ThemeEntry.

    //ThemeInterface
    AotThemeInterface = _Theme | _Subcategory2, ///< "Theme/ThemeInterface" subcategory flag.
    AotThemeEntryInterface,                     ///< ThemeInterface - Theme entry interface.
    AotAnchorInterface,                         ///< ThemeInterface - Theme anchor interface.
    AotCompositePropertyInterface,              ///< ThemeInterface - Composite property interface.

    AotAppearanceCollectionThemeEntryInterface, ///< ThemeInterface - AppearanceCollectionThemeEntryInterface.
    AotAppearanceThemeEntryInterface,           ///< ThemeInterface - AppearanceThemeEntryInterface.
    AotBitmapThemeEntryInterface,               ///< ThemeInterface - BitmapThemeEntryInterfac.
    AotComposite2DThemeEntryInterface,          ///< ThemeInterface - Composite2DThemeEntryInterface.
    AotCompositeThemeEntryInterface,            ///< ThemeInterface - CompositeThemeEntryInterface.
    AotFontThemeEntryInterface,                 ///< ThemeInterface - FontThemeEntryInterface.
    AotMaterialThemeEntryInterface,             ///< ThemeInterface - MaterialThemeEntryInterface.
    AotRawResourceThemeEntryInterface,          ///< ThemeInterface - RawResourceThemeEntryInterface.
    AotRenderModeThemeEntryInterface,           ///< ThemeInterface - RenderModeThemeEntryInterface.
    AotTextureThemeEntryInterface,              ///< ThemeInterface - TextureThemeEntryInterface.
    AotUniformSetterThemeEntryInterface,        ///< ThemeInterface - UniformSetterThemeEntryInterface.

    _MiscDummy = _Misc | _Subcategory1, ///< "Miscellaneous/Dummy" subcategory flag.
    AotReferencedTemplate,              ///< Misc - ReferencedTemplate.
    AotTransition,                      ///< Misc - Transition.
    AotFolder,                          ///< Misc - Folder.
    AotImportFolder,                    ///< Misc - ImportFolder.
    AotTransitionRule,                  ///< Misc - TransitionRule.
    AotTransitionHint,                  ///< Misc - TransitionHint.
    AotTransitionConfiguration,         ///< Misc   TransitionConfiguration

    _Platform = _Misc | _Subcategory2, ///< "Miscellaneous/Platform" subcategory flag.
    AotDevicePackage,                  ///< Platform - Device package.
    AotDisplay,                        ///< Platform - Display.
    AotGraphicDeviceUnit,              ///< Platform - GDU.
    AotContextResourcePool,            ///< Platform - Context resource pool.
    AotTargetSystem,                   ///< Platform - Target system.

    _Resources = _Misc | _Subcategory3, ///< "Miscellaneous/Resources" subcategory flag.
    AotRawResource,                     ///< Resources - RawResource.
    AotFont,                            ///< Resources - Font.
    AotShader,                          ///< Resources - Shader.
    AotLanguage,                        ///< Resources - Language.
    AotScript,                          ///< Resources - Script.
    AotScriptComponent,                 ///< Resources - ScriptComponent.
    AotBitmap = 7,                      ///< Resources - Bitmap (compatibility with old solutions)
    AotVertexBuffer = 27,               ///< Resources - VertexBuffer (compatibility with old solutions)

    _Config = _Misc | _Subcategory4,    ///< "Miscellaneous/Config" subcategory flag.
    AotConfig,                          ///< Config - Config.
    AotFilterItem,                      ///< Config - Filter item.
    AotColorInfo,                       ///< Config - Color info.
    AotFontInfo,                        ///< Config - Font info.
    AotDefaultShaderPrograms,           ///< Config - Default shader programs.
    AotDynamicItemDefinition,           ///< Config - Dynamic item definition.
    AotLayoutDefinition,                ///< Config - Layout definition.
    AotGizmo,                           ///< Config - Gizmo.
    AotScope,                           ///< Config - Scope.
    AotShaderConfig,                    ///< Config - Shader config.
    AotAssetProfile,                    ///< Config - Asset profile.
    AotEditorProfile,                   ///< Config - Editor profile.
    AotIcon,                            ///< Config - Icon.
    AotTextStyleEntry,                  ///< Config - Text style entry.
    AotTextStyle,                       ///< Config - Text style.
    AotCultureInfo,                     ///< Config - Culture info.

    _StateMachine = _Misc | _Subcategory7, ///< "Miscellaneous/StateMachine" subcategory flag.
    AotStateMachine,                       ///< StateMachine - StateMachine.
    AotStateMachineItem,                   ///< StateMachine - StateMachine item.
    AotStateMachineItems,                  ///< StateMachine - StateMachine items.
    AotState,                              ///< StateMachine - State.
    AotStateTransition,                    ///< StateMachine - State transition.
    AotInitialState,                       ///< StateMachine - InitialState.
    AotHistoryState,                       ///< StateMachine - HistoryState.
    AotSubChartState,                      ///< StateMachine - SubChartState.
    AotStateMachineUIContent,              ///< StateMachine - StateMachine UIContent.

    AotCollectionItem = _Collection,                                            ///< Collection item.
    AotAnchorDefinitionCollection = AotAnchorDefinition | _Collection,          ///< Collection - AotAnchorDefinition collection.
    AotAnchorInterfaceCollection = AotAnchorInterface | _Collection,            ///< Collection - AotAnchorInterface collection.
    AotAppearanceCollection = AotAppearance | _Collection,                      ///< Collection - AotAppearance collection.
    AotCompositePropertyDefinitionCollection = AotCompositePropertyDefinition | _Collection,    ///< Collection - AotCompositePropertyDefinition collection.
    AotCompositePropertyInterfaceCollection = AotCompositePropertyInterface | _Collection,  ///< Collection - AotCompositePropertyInterface collection.
    AotEffect2DCollection = AotEffect2D | _Collection,                          ///< Collection - AotEffect2D collection.
    AotMorphItemCollection = AotMorphItem | _Collection,                        ///< Collection - AotMorphItem collection.
    AotRenderOrderBinCollection = AotRenderOrderBin | _Collection,              ///< Collection - AotRenderOrderBin collection.
    AotTextureCollection = AotTexture | _Collection,                            ///< Collection - AotTexture collection.
    AotThemeEntryCollection = AotThemeEntry | _Collection,                      ///< Collection - AotThemeEntry collection.
    AotThemeEntryInterfaceCollection = AotThemeEntryInterface | _Collection,    ///< Collection - AotThemeEntryInterface collection.
    AotWidgetCollection = AotWidget | _Collection,                              ///< Collection - AotWidget collection.
    AotTransitionRuleCollection = AotTransitionRule | _Collection,              ///< Collection - AotTransitionRule Collection.
    AotTransitionCollection = AotTransition | _Collection,                      ///< Collection - AotTransition Collection.
    AotBehaviorCollection = AotBehavior | _Collection                           ///< Collection - AotBehavior collection.
};

/** @} */ // end of AssetLoaderBase

#endif // CANDERA_ASSETLIBENTITIES_H
